package ru.yandex.intranet.d.web.converters

import ru.yandex.intranet.d.i18n.Locales
import ru.yandex.intranet.d.model.resources.ResourceModel
import ru.yandex.intranet.d.model.resources.segmentations.ResourceSegmentationModel
import ru.yandex.intranet.d.model.resources.segmentations.SegmentationUISettings
import ru.yandex.intranet.d.model.resources.segments.ResourceSegmentModel
import ru.yandex.intranet.d.model.resources.types.ResourceTypeModel
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel
import ru.yandex.intranet.d.services.resources.ExpandedResources
import ru.yandex.intranet.d.services.resources.SelectionResourceTreeNode
import ru.yandex.intranet.d.services.resources.SelectionResourceTreeNode.EMPTY_SEGMENT_PLACEHOLDER
import ru.yandex.intranet.d.services.resources.SelectionTreeResourceDto
import ru.yandex.intranet.d.web.model.ResourceDto
import ru.yandex.intranet.d.web.model.resources.SelectionResourceTreeNodeDto
import ru.yandex.intranet.d.web.model.resources.directory.segmentations.SegmentationUISettingsDto
import ru.yandex.intranet.d.web.model.units.front.FrontUnitsEnsembleDto
import java.util.*
import java.util.stream.Collectors


/**
 *
 * @author Mustakayev Marat <mmarat248@yandex-team.ru>
 */

fun toSelectionResourceTreeNodeDto(
    t: ExpandedResources<SelectionResourceTreeNode>, locale: Locale): SelectionResourceTreeNodeDto =
    toDto(
        t.resources, locale, t.resourceTypes, t.unitsEnsembles,
        t.resourceSegmentations, t.resourceSegments, null, true
    )


private fun toDto(
    root: SelectionResourceTreeNode,
    locale: Locale,
    resourceTypes: Map<String, ResourceTypeModel>,
    unitsEnsemblesById: Map<String, UnitsEnsembleModel>,
    segmentationsById: Map<String, ResourceSegmentationModel>,
    segmentsById: Map<String, ResourceSegmentModel>,
    currentNodeSegmentId: String?,
    rootNode: Boolean
): SelectionResourceTreeNodeDto {
    if (root.resources != null) {
        return SelectionResourceTreeNodeDto.builder()
            .accountsSpacesIds(root.resources.stream()
                .map { obj: ResourceModel -> obj.accountsSpacesId }
                .filter(Objects::nonNull)
                .collect(Collectors.toSet()))
            .resources(
                getResourceDto(
                    root.resources, locale, resourceTypes, unitsEnsemblesById,
                    segmentationsById, segmentsById
                )
            )
            .currentNodeSegmentId(currentNodeSegmentId)
            .currentNodeSegmentKey(currentNodeSegmentId?.let { segmentsById[it]?.key })
            .currentNodeSegmentUncommon(currentNodeSegmentId?.let { segmentsById[it]?.uncommon?.orElse(false) })
            .build()
    }
    if (root.childrenBySegmentId == null) {
        return SelectionResourceTreeNodeDto.empty()
    }
    val childrenDtoBySegmentId: MutableMap<String, SelectionResourceTreeNodeDto?> = HashMap()
    root.childrenBySegmentId.forEach { (segmentId, node) ->
        childrenDtoBySegmentId[
            (
                if (segmentId.equals(EMPTY_SEGMENT_PLACEHOLDER))
                    segmentId
                else
                    getName(segmentsById[segmentId]!!, locale)
            )
        ] = toDto(
                node, locale, resourceTypes, unitsEnsemblesById, segmentationsById, segmentsById,
                if (EMPTY_SEGMENT_PLACEHOLDER == segmentId) null else segmentId, false
            )
    }
    var uiSettings: SegmentationUISettingsDto? = null
    var segmentationKey: String? = null
    if (root.segmentationId != null) {
        val segmentation = segmentationsById[root.segmentationId]
        if (segmentation != null) {
            uiSettings = segmentation.uiSettings.map { model: SegmentationUISettings? ->
                SegmentationUISettingsDto(
                    model!!
                )
            }.orElse(null)
            segmentationKey = segmentation.key
        }
    }
    val builder = SelectionResourceTreeNodeDto.builder()
        .childrenBySegmentId(childrenDtoBySegmentId)
        .segmentationId(root.segmentationId)
        .segmentationKey(segmentationKey)
        .segmentationUISettings(uiSettings)
        .currentNodeSegmentId(currentNodeSegmentId)
        .currentNodeSegmentKey(currentNodeSegmentId?.let { segmentsById[it]?.key })
        .currentNodeSegmentUncommon(currentNodeSegmentId?.let { segmentsById[it]?.uncommon?.orElse(false) })
    if (rootNode) {
        builder.unitsEnsemblesById(unitsEnsemblesById.mapValues { FrontUnitsEnsembleDto(it.value, locale) })
    }
    return builder.build()
}

private fun getName(model: ResourceSegmentModel, locale: Locale): String {
    return Locales.select(model.nameEn, model.nameRu, locale)
}


private fun getResourceDto(
    resourceModel: ResourceModel, locale: Locale,
    unitsEnsemblesById: Map<String, UnitsEnsembleModel>,
    segmentationsById: Map<String, ResourceSegmentationModel>,
    segmentsById: Map<String, ResourceSegmentModel>
): ResourceDto {
    return ResourceDto(resourceModel, locale, unitsEnsemblesById, segmentationsById, segmentsById)
}

private fun getResourceDto(
    resourceModel: List<ResourceModel>,
    locale: Locale,
    resourceTypes: Map<String, ResourceTypeModel>,
    unitsEnsemblesById: Map<String, UnitsEnsembleModel>,
    segmentationsById: Map<String, ResourceSegmentationModel>,
    segmentsById: Map<String, ResourceSegmentModel>
): List<SelectionTreeResourceDto> {
    return resourceModel.stream()
        .map { resource: ResourceModel ->
            val resourceType = resourceTypes[resource.resourceTypeId]!!
            SelectionTreeResourceDto(
                getResourceDto(resource, locale, unitsEnsemblesById, segmentationsById, segmentsById),
                resourceType.nameEn,
                resourceType.nameRu,
                resourceType.sortingOrder,
                resourceType.key,
                resource.key
            )
        }
        .collect(Collectors.toList())
}
