package ru.yandex.intranet.d.web.errors;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.google.protobuf.Any;
import com.google.rpc.Code;
import com.google.rpc.Status;
import io.grpc.StatusRuntimeException;
import io.grpc.protobuf.StatusProto;
import org.springframework.context.MessageSource;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import ru.yandex.intranet.d.backend.service.proto.ErrorDetails;
import ru.yandex.intranet.d.backend.service.proto.FieldError;
import ru.yandex.intranet.d.model.accounts.OperationErrorKind;
import ru.yandex.intranet.d.services.integration.providers.ProviderError;
import ru.yandex.intranet.d.services.integration.providers.rest.model.ErrorMessagesDto;
import ru.yandex.intranet.d.util.result.ErrorCollection;
import ru.yandex.intranet.d.util.result.ErrorType;
import ru.yandex.intranet.d.util.result.TypedError;
import ru.yandex.intranet.d.web.model.ErrorCollectionDto;

/**
 * Error response utils.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public final class Errors {

    private Errors() {
    }

    public static ResponseEntity<ErrorCollectionDto> toResponse(ErrorCollection errorCollection) {
        Set<ErrorType> errorTypes = new HashSet<>();
        errorCollection.getErrors().forEach(e -> errorTypes.add(e.getType()));
        errorCollection.getFieldErrors().values().forEach(v -> v.forEach(e -> errorTypes.add(e.getType())));
        HttpStatus responseStatus = prepareResponseStatus(errorTypes);
        ErrorCollection errorCollectionToReturn = errorCollection;
        if (errorTypes.contains(ErrorType.FORBIDDEN)) {
            errorCollectionToReturn = filterForbidden(errorCollection);
        }
        ErrorCollectionDto dto = toDto(errorCollectionToReturn);
        return ResponseEntity.status(responseStatus).contentType(MediaType.APPLICATION_JSON).body(dto);
    }

    public static StatusRuntimeException toGrpcError(ErrorCollection errorCollection,
                                                     MessageSource messages, Locale locale) {
        Set<ErrorType> errorTypes = new HashSet<>();
        errorCollection.getErrors().forEach(e -> errorTypes.add(e.getType()));
        errorCollection.getFieldErrors().values().forEach(v -> v.forEach(e -> errorTypes.add(e.getType())));
        Code grpcCode = prepareGrpcCode(errorTypes);
        ErrorCollection errorCollectionToReturn = errorCollection;
        if (errorTypes.contains(ErrorType.FORBIDDEN)) {
            errorCollectionToReturn = filterForbidden(errorCollection);
        }
        Status status = prepareGrpcStatus(grpcCode, errorCollectionToReturn, messages, locale);
        return StatusProto.toStatusRuntimeException(status);
    }

    public static ErrorCollectionDto toDto(ErrorCollection errorCollection) {
        Set<String> errors = errorCollection.getErrors().stream()
                .map(TypedError::getError).collect(Collectors.toSet());
        Map<String, Set<String>> fieldErrors = new HashMap<>();
        errorCollection.getFieldErrors().forEach((k, s) -> s
                .forEach(e -> fieldErrors.computeIfAbsent(k, v -> new HashSet<>()).add(e.getError())));
        return new ErrorCollectionDto(errors, fieldErrors, errorCollection.getDetails());
    }

    public static ErrorCollectionDto toDto(List<ErrorCollectionDto> errorCollectionDto) {
        Set<String> errors = new HashSet<>();
        Map<String, Set<String>> fieldErrors = new HashMap<>();
        Map<String, Set<Object>> details = new HashMap<>();
        for (ErrorCollectionDto dto : errorCollectionDto) {
            errors.addAll(dto.getErrors());
            dto.getFieldErrors().forEach((k, s) -> s
                    .forEach(e -> fieldErrors.computeIfAbsent(k, v -> new HashSet<>()).add(e)));
            dto.getDetails().forEach((k, s) -> s
                    .forEach(d -> details.computeIfAbsent(k, v -> new HashSet<>()).add(d)));
        }
        return new ErrorCollectionDto(errors, fieldErrors, details);
    }

    public static String flattenErrors(ErrorCollection errors) {
        Stream<String> simpleErrors = errors.getErrors().stream().map(TypedError::getError);
        Stream<String> fieldErrors = errors.getFieldErrors().entrySet().stream().flatMap(e -> e.getValue().stream()
                .map(v -> e.getKey() + ": " + v.getError()));
        return Stream.concat(simpleErrors, fieldErrors).collect(Collectors.joining("\n"));
    }

    public static String flattenErrors(ErrorMessagesDto errors) {
        Stream<String> simpleErrors = errors.getMessage().stream();
        Stream<String> fieldErrors = errors.getFieldErrors().orElse(Map.of()).entrySet().stream()
                .map(e -> e.getKey() + ": " + e.getValue());
        return Stream.concat(simpleErrors, fieldErrors).collect(Collectors.joining("\n"));
    }

    public static String flattenErrors(String message, Map<String, String> badRequestDetails) {
        Stream<String> simpleErrors = Optional.ofNullable(message).stream();
        Stream<String> fieldErrors = badRequestDetails.entrySet().stream().map(e -> e.getKey() + ": " + e.getValue());
        return Stream.concat(simpleErrors, fieldErrors).collect(Collectors.joining("\n"));
    }

    public static String flattenProviderErrorResponse(ProviderError error, String prefix) {
        return error.match(new ProviderError.Cases<>() {
            @Override
            public String httpError(int statusCode) {
                HttpStatus resolvedStatus = HttpStatus.resolve(statusCode);
                String statusMessage = "" + statusCode
                        + (resolvedStatus != null ? " " + resolvedStatus.getReasonPhrase() : "");
                return (prefix != null ? prefix + " " : "") + statusMessage;
            }

            @Override
            public String httpExtendedError(int statusCode, ErrorMessagesDto errors) {
                HttpStatus resolvedStatus = HttpStatus.resolve(statusCode);
                String statusMessage = "" + statusCode
                        + (resolvedStatus != null ? " " + resolvedStatus.getReasonPhrase() : "");
                String flatErrors = flattenErrors(errors);
                return (prefix != null ? prefix + " " : "") + statusMessage
                        + (flatErrors.isEmpty() ? "" : " " + flatErrors);
            }

            @Override
            public String grpcError(io.grpc.Status.Code statusCode, String message) {
                String statusMessage = statusCode.name() + (message == null || message.isEmpty() ? "" : " " + message);
                return (prefix != null ? prefix + " " : "") + statusMessage;
            }

            @Override
            public String grpcExtendedError(io.grpc.Status.Code statusCode, String message,
                                            Map<String, String> badRequestDetails) {
                String statusMessage = statusCode.name() + (message == null || message.isEmpty() ? "" : " " + message);
                String flatErrors = flattenErrors(message, badRequestDetails);
                return (prefix != null ? prefix + " " : "") + statusMessage
                        + (flatErrors.isEmpty() ? "" : " " + flatErrors);
            }
        });
    }

    public static ErrorCollection providerErrorResponseToErrorCollection(ProviderError error) {
        return error.match(new ProviderError.Cases<>() {
            @Override
            public ErrorCollection httpError(int statusCode) {
                HttpStatus resolvedStatus = HttpStatus.resolve(statusCode);
                String statusMessage = "" + statusCode
                        + (resolvedStatus != null ? " " + resolvedStatus.getReasonPhrase() : "");
                Function<String, TypedError> typeWrapper = getTypeWrapper(statusCode);
                return ErrorCollection.builder().addError(typeWrapper.apply(statusMessage)).build();
            }

            @Override
            public ErrorCollection httpExtendedError(int statusCode, ErrorMessagesDto errors) {
                HttpStatus resolvedStatus = HttpStatus.resolve(statusCode);
                String statusMessage = "" + statusCode
                        + (resolvedStatus != null ? " " + resolvedStatus.getReasonPhrase() : "");
                Function<String, TypedError> typeWrapper = getTypeWrapper(statusCode);
                ErrorCollection.Builder errorsBuilder = ErrorCollection.builder();
                errorsBuilder.addError(typeWrapper.apply(statusMessage));
                errors.getMessage().ifPresent(e -> errorsBuilder.addError(typeWrapper.apply(e)));
                errors.getFieldErrors().ifPresent(e -> e.forEach((k, v) -> errorsBuilder
                        .addError(k, typeWrapper.apply(v))));
                return errorsBuilder.build();
            }

            @Override
            public ErrorCollection grpcError(io.grpc.Status.Code statusCode, String message) {
                String statusMessage = statusCode.name() + (message == null || message.isEmpty() ? "" : " " + message);
                Function<String, TypedError> typeWrapper = getTypeWrapper(statusCode);
                return ErrorCollection.builder().addError(typeWrapper.apply(statusMessage)).build();
            }

            @Override
            public ErrorCollection grpcExtendedError(io.grpc.Status.Code statusCode, String message,
                                                     Map<String, String> badRequestDetails) {
                String statusMessage = statusCode.name() + (message == null || message.isEmpty() ? "" : " " + message);
                Function<String, TypedError> typeWrapper = getTypeWrapper(statusCode);
                ErrorCollection.Builder errorsBuilder = ErrorCollection.builder();
                errorsBuilder.addError(typeWrapper.apply(statusMessage));
                badRequestDetails.forEach((k, v) -> errorsBuilder.addError(k, typeWrapper.apply(v)));
                return errorsBuilder.build();
            }
        });
    }

    public static OperationErrorKind providerErrorResponseToOperationErrorKind(ProviderError error) {
        return error.match(new ProviderError.Cases<>() {
            @Override
            public OperationErrorKind httpError(int statusCode) {
                if (statusCode == HttpStatus.CONFLICT.value()) {
                    return OperationErrorKind.ALREADY_EXISTS;
                } else if (statusCode == HttpStatus.PRECONDITION_FAILED.value()) {
                    return OperationErrorKind.FAILED_PRECONDITION;
                } else if (ProviderError.isRetryableCode(statusCode)) {
                    return OperationErrorKind.UNAVAILABLE;
                } else {
                    return OperationErrorKind.INVALID_ARGUMENT;
                }
            }

            @Override
            public OperationErrorKind httpExtendedError(int statusCode, ErrorMessagesDto errors) {
                if (statusCode == HttpStatus.CONFLICT.value()) {
                    return OperationErrorKind.ALREADY_EXISTS;
                } else if (statusCode == HttpStatus.PRECONDITION_FAILED.value()) {
                    return OperationErrorKind.FAILED_PRECONDITION;
                } else if (ProviderError.isRetryableCode(statusCode)) {
                    return OperationErrorKind.UNAVAILABLE;
                } else {
                    return OperationErrorKind.INVALID_ARGUMENT;
                }
            }

            @Override
            public OperationErrorKind grpcError(io.grpc.Status.Code statusCode, String message) {
                if (statusCode == io.grpc.Status.Code.ALREADY_EXISTS) {
                    return OperationErrorKind.ALREADY_EXISTS;
                } else if (statusCode == io.grpc.Status.Code.FAILED_PRECONDITION) {
                    return OperationErrorKind.FAILED_PRECONDITION;
                } else if (ProviderError.isRetryableCode(statusCode)) {
                    return OperationErrorKind.UNAVAILABLE;
                } else {
                    return OperationErrorKind.INVALID_ARGUMENT;
                }
            }

            @Override
            public OperationErrorKind grpcExtendedError(io.grpc.Status.Code statusCode, String message,
                                                     Map<String, String> badRequestDetails) {
                if (statusCode == io.grpc.Status.Code.ALREADY_EXISTS) {
                    return OperationErrorKind.ALREADY_EXISTS;
                } else if (statusCode == io.grpc.Status.Code.FAILED_PRECONDITION) {
                    return OperationErrorKind.FAILED_PRECONDITION;
                } else if (ProviderError.isRetryableCode(statusCode)) {
                    return OperationErrorKind.UNAVAILABLE;
                } else {
                    return OperationErrorKind.INVALID_ARGUMENT;
                }
            }
        });
    }

    private static Function<String, TypedError> getTypeWrapper(int statusCode) {
        if (statusCode == HttpStatus.CONFLICT.value()) {
            return TypedError::conflict;
        } else if (statusCode == HttpStatus.PRECONDITION_FAILED.value()) {
            return TypedError::versionMismatch;
        } else if (ProviderError.isRetryableCode(statusCode)) {
            return TypedError::unavailable;
        } else {
            return TypedError::badRequest;
        }
    }

    private static Function<String, TypedError> getTypeWrapper(io.grpc.Status.Code statusCode) {
        if (statusCode == io.grpc.Status.Code.ALREADY_EXISTS) {
            return TypedError::conflict;
        } else if (statusCode == io.grpc.Status.Code.FAILED_PRECONDITION) {
            return TypedError::versionMismatch;
        } else if (ProviderError.isRetryableCode(statusCode)) {
            return TypedError::unavailable;
        } else {
            return TypedError::badRequest;
        }
    }

    private static Status prepareGrpcStatus(Code grpcCode, ErrorCollection errorCollection,
                                            MessageSource messages, Locale locale) {
        return Status.newBuilder()
                .setCode(grpcCode.getNumber())
                .setMessage(codeToMessage(grpcCode, messages, locale) + (errorCollection.hasAnyErrors()
                        ? "\n" + flattenErrors(errorCollection) : ""))
                .addDetails(toDetails(errorCollection))
                .build();
    }

    private static String codeToMessage(Code grpcCode, MessageSource messages, Locale locale) {
        switch (grpcCode) {
            case OK:
                return messages.getMessage("errors.grpc.code.ok", null, locale);
            case CANCELLED:
                return messages.getMessage("errors.grpc.code.cancelled", null, locale);
            case UNKNOWN:
                return messages.getMessage("errors.grpc.code.unknown", null, locale);
            case INVALID_ARGUMENT:
                return messages.getMessage("errors.grpc.code.invalid.argument", null, locale);
            case DEADLINE_EXCEEDED:
                return messages.getMessage("errors.grpc.code.deadline.exceeded", null, locale);
            case NOT_FOUND:
                return messages.getMessage("errors.grpc.code.not.found", null, locale);
            case ALREADY_EXISTS:
                return messages.getMessage("errors.grpc.code.already.exists", null, locale);
            case PERMISSION_DENIED:
                return messages.getMessage("errors.grpc.code.permission.denied", null, locale);
            case UNAUTHENTICATED:
                return messages.getMessage("errors.grpc.code.unauthenticated", null, locale);
            case RESOURCE_EXHAUSTED:
                return messages.getMessage("errors.grpc.code.resource.exhausted", null, locale);
            case FAILED_PRECONDITION:
                return messages.getMessage("errors.grpc.code.failed.precondition", null, locale);
            case ABORTED:
                return messages.getMessage("errors.grpc.code.aborted", null, locale);
            case OUT_OF_RANGE:
                return messages.getMessage("errors.grpc.code.out.of.range", null, locale);
            case UNIMPLEMENTED:
                return messages.getMessage("errors.grpc.code.unimplemented", null, locale);
            case INTERNAL:
                return messages.getMessage("errors.grpc.code.internal", null, locale);
            case UNAVAILABLE:
                return messages.getMessage("errors.grpc.code.unavailable", null, locale);
            case DATA_LOSS:
                return messages.getMessage("errors.grpc.code.data.loss", null, locale);
            default:
                return messages.getMessage("errors.grpc.code.unknown", null, locale);
        }
    }

    private static Any toDetails(ErrorCollection errorCollection) {
        ErrorDetails.Builder result = ErrorDetails.newBuilder();
        errorCollection.getErrors().forEach(e -> result.addErrors(e.getError()));
        errorCollection.getFieldErrors().forEach((k, v) -> {
            FieldError.Builder fieldError = FieldError.newBuilder();
            fieldError.setKey(k);
            v.forEach(e -> fieldError.addErrors(e.getError()));
            result.addFieldErrors(fieldError.build());
        });
        return Any.pack(result.build());
    }

    private static HttpStatus prepareResponseStatus(Set<ErrorType> errorTypes) {
        if (errorTypes.contains(ErrorType.FORBIDDEN)) {
            return HttpStatus.FORBIDDEN;
        } else if (errorTypes.contains(ErrorType.CONFLICT)) {
            return HttpStatus.CONFLICT;
        } else if (errorTypes.contains(ErrorType.VERSION_MISMATCH)) {
            return HttpStatus.PRECONDITION_FAILED;
        } else if (errorTypes.contains(ErrorType.LOCKED)) {
            return HttpStatus.LOCKED;
        } else if (errorTypes.contains(ErrorType.NOT_FOUND)) {
            return HttpStatus.NOT_FOUND;
        } else if (errorTypes.contains(ErrorType.BAD_REQUEST)) {
            return HttpStatus.BAD_REQUEST;
        } else if (errorTypes.contains(ErrorType.INVALID)) {
            return HttpStatus.UNPROCESSABLE_ENTITY;
        } else if (errorTypes.contains(ErrorType.TOO_MANY_REQUESTS)) {
            return HttpStatus.TOO_MANY_REQUESTS;
        } else if (errorTypes.contains(ErrorType.UNEXPECTED)) {
            return HttpStatus.INTERNAL_SERVER_ERROR;
        } else if (errorTypes.contains(ErrorType.UNAVAILABLE)) {
            return HttpStatus.SERVICE_UNAVAILABLE;
        } else {
            return HttpStatus.BAD_REQUEST;
        }
    }

    private static Code prepareGrpcCode(Set<ErrorType> errorTypes) {
        if (errorTypes.contains(ErrorType.FORBIDDEN)) {
            return Code.PERMISSION_DENIED;
        } else if (errorTypes.contains(ErrorType.CONFLICT)) {
            return Code.ALREADY_EXISTS;
        } else if (errorTypes.contains(ErrorType.VERSION_MISMATCH)) {
            return Code.FAILED_PRECONDITION;
        } else if (errorTypes.contains(ErrorType.LOCKED)) {
            return Code.UNAVAILABLE;
        } else if (errorTypes.contains(ErrorType.NOT_FOUND)) {
            return Code.NOT_FOUND;
        } else if (errorTypes.contains(ErrorType.BAD_REQUEST)) {
            return Code.INVALID_ARGUMENT;
        } else if (errorTypes.contains(ErrorType.INVALID)) {
            return Code.INVALID_ARGUMENT;
        } else if (errorTypes.contains(ErrorType.TOO_MANY_REQUESTS)) {
            return Code.UNAVAILABLE;
        } else if (errorTypes.contains(ErrorType.UNEXPECTED)) {
            return Code.INTERNAL;
        } else if (errorTypes.contains(ErrorType.UNAVAILABLE)) {
            return Code.UNAVAILABLE;
        } else {
            return Code.INVALID_ARGUMENT;
        }
    }

    private static ErrorCollection filterForbidden(ErrorCollection errorCollection) {
        ErrorCollection.Builder builder = ErrorCollection.builder();
        errorCollection.getErrors().forEach(e -> {
            if (e.getType() == ErrorType.FORBIDDEN) {
                builder.addError(e);
            }
        });
        errorCollection.getFieldErrors().forEach((k, s) -> s.forEach(e -> {
            if (e.getType() == ErrorType.FORBIDDEN) {
                builder.addError(k, e);
            }
        }));
        return builder.build();
    }

}
