package ru.yandex.intranet.d.web.jackson;

import java.io.IOException;
import java.math.BigDecimal;
import java.util.Map;

import javax.annotation.Nullable;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectReader;
import com.fasterxml.jackson.databind.exc.InvalidDefinitionException;
import com.fasterxml.jackson.databind.util.TokenBuffer;
import org.reactivestreams.Publisher;
import org.springframework.core.MethodParameter;
import org.springframework.core.ResolvableType;
import org.springframework.core.codec.CodecException;
import org.springframework.core.codec.DecodingException;
import org.springframework.core.codec.Hints;
import org.springframework.core.io.buffer.DataBuffer;
import org.springframework.core.io.buffer.DataBufferLimitException;
import org.springframework.core.io.buffer.DataBufferUtils;
import org.springframework.core.log.LogFormatUtils;
import org.springframework.http.codec.json.Jackson2CodecSupport;
import org.springframework.http.codec.json.Jackson2JsonDecoder;
import org.springframework.lang.NonNull;
import org.springframework.util.Assert;
import org.springframework.util.MimeType;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;
/**
 * Replacement for {@link Jackson2JsonDecoder}
 * used to replace an exception for a large requests
 * to provide a status code 413 instead of 500
 *
 * @author Petr Surkov <petrsurkov@yandex-team.ru>
 */
@SuppressWarnings({"checkstyle:RightCurly", "checkstyle:LineLength"})
public class CustomJackson2Decoder extends Jackson2JsonDecoder {
    private static final String ACTUAL_TYPE_HINT = Jackson2CodecSupport.class.getName() + ".actualType";

    public CustomJackson2Decoder(ObjectMapper mapper, MimeType... mimeTypes) {
        super(mapper, mimeTypes);
    }

    @Override
    @NonNull
    public Flux<Object> decode(@NonNull Publisher<DataBuffer> input, @NonNull ResolvableType elementType,
                               @Nullable MimeType mimeType, @Nullable Map<String, Object> hints) {
        ObjectMapper mapper = selectObjectMapper(elementType, mimeType);
        if (mapper == null) {
            throw new IllegalStateException("No ObjectMapper for " + elementType);
        }

        boolean forceUseOfBigDecimal = mapper.isEnabled(DeserializationFeature.USE_BIG_DECIMAL_FOR_FLOATS);
        if (BigDecimal.class.equals(elementType.getType())) {
            forceUseOfBigDecimal = true;
        }

        Flux<DataBuffer> processed = processInput(input, elementType, mimeType, hints);
        Flux<TokenBuffer> tokens = CustomJackson2Tokenizer.tokenize(processed, mapper.getFactory(), mapper,
                true, forceUseOfBigDecimal, getMaxInMemorySize());

        ObjectReader reader = getObjectReader(mapper, elementType, hints);

        return tokens.handle((tokenBuffer, sink) -> {
            try {
                Object value = reader.readValue(tokenBuffer.asParser(mapper));
                logValue(value, hints);
                if (value != null) {
                    sink.next(value);
                }
            }
            catch (IOException ex) {
                sink.error(processException(ex));
            }
        });
    }

    @Override
    public Mono<Object> decodeToMono(Publisher<DataBuffer> input, ResolvableType elementType,
                                     @Nullable MimeType mimeType, @Nullable Map<String, Object> hints) {
        return DataBufferUtils.join(input, getMaxInMemorySize())
                .onErrorMap(throwable -> {
                    if (throwable instanceof DataBufferLimitException) {
                        return new PayloadTooLargeException(
                                "Payload content length greater than maximum allowed " + getMaxInMemorySize());
                    }
                    return throwable;
                })
                .flatMap(dataBuffer -> Mono.justOrEmpty(decode(dataBuffer, elementType, mimeType, hints)));
    }

    private ObjectReader getObjectReader(ObjectMapper mapper, ResolvableType elementType,
                                         @Nullable Map<String, Object> hints) {
        Assert.notNull(elementType, "'elementType' must not be null");
        Class<?> contextClass = getContextClass(elementType);
        if (contextClass == null && hints != null) {
            contextClass = getContextClass((ResolvableType) hints.get(ACTUAL_TYPE_HINT));
        }
        JavaType javaType = getJavaType(elementType.getType(), contextClass);
        Class<?> jsonView = (hints != null ? (Class<?>) hints.get(Jackson2CodecSupport.JSON_VIEW_HINT) : null);
        return jsonView != null ?
                mapper.readerWithView(jsonView).forType(javaType) :
                mapper.readerFor(javaType);
    }

    @Nullable
    private Class<?> getContextClass(@Nullable ResolvableType elementType) {
        MethodParameter param = (elementType != null ? getParameter(elementType)  : null);
        return (param != null ? param.getContainingClass() : null);
    }

    private CodecException processException(IOException ex) {
        if (ex instanceof InvalidDefinitionException) {
            JavaType type = ((InvalidDefinitionException) ex).getType();
            return new CodecException("Type definition error: " + type, ex);
        }
        if (ex instanceof JsonProcessingException) {
            String originalMessage = ((JsonProcessingException) ex).getOriginalMessage();
            return new DecodingException("JSON decoding error: " + originalMessage, ex);
        }
        return new DecodingException("I/O error while parsing input stream", ex);
    }

    private void logValue(@Nullable Object value, @Nullable Map<String, Object> hints) {
        if (!Hints.isLoggingSuppressed(hints)) {
            LogFormatUtils.traceDebug(logger, traceOn -> {
                String formatted = LogFormatUtils.formatValue(value, !traceOn);
                return Hints.getLogPrefix(hints) + "Decoded [" + formatted + "]";
            });
        }
    }
}
