package ru.yandex.intranet.d.web.model;

import java.util.Objects;
import java.util.Optional;

import javax.annotation.Nullable;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import io.swagger.v3.oas.annotations.media.Schema;

import ru.yandex.intranet.d.model.accounts.AccountModel;
import ru.yandex.intranet.d.util.DisplayUtil;
import ru.yandex.intranet.d.web.model.accounts.AccountReserveTypeDto;

/**
 * Account DTO.
 *
 * @author Nikita Minin <spasitel@yandex-team.ru>
 */
@Schema(description = "Account.")
@JsonIgnoreProperties(ignoreUnknown = true)
public final class AccountDto {
    private final String id;
    private final String displayName;
    private final String folderId;
    private final boolean deleted;
    private final String accountsSpacesId;
    private final boolean freeTier;
    private final String providerId;
    @Nullable
    private final AccountReserveTypeDto reserveType;

    @SuppressWarnings("ParameterNumber")
    @JsonCreator
    public AccountDto(
            String id,
            String displayName,
            String folderId,
            boolean deleted,
            String accountsSpacesId,
            boolean freeTier,
            String providerId,
            @Nullable AccountReserveTypeDto reserveType
    ) {
        this.id = id;
        this.displayName = displayName;
        this.folderId = folderId;
        this.deleted = deleted;
        this.accountsSpacesId = accountsSpacesId;
        this.freeTier = freeTier;
        this.providerId = providerId;
        this.reserveType = reserveType;
    }

    public static AccountDto fromModel(AccountModel account) {
        return new AccountDto(
                account.getId(), // id
                DisplayUtil.getAccountDisplayString(account), // displayName
                account.getFolderId(), // folderId
                account.isDeleted(), // deleted
                account.getAccountsSpacesId().orElse(null), // accountsSpacesId
                account.isFreeTier(), // freeTier
                account.getProviderId(), // providerId
                AccountReserveTypeDto.fromModel(account.getReserveType().orElse(null)) // reserveType
        );
    }

    @Schema(description = "Unique id of account.", required = true)
    public String getId() {
        return id;
    }

    @Schema(description = "Display name, or key, or id of account.", required = true)
    public String getDisplayName() {
        return displayName;
    }

    @Schema(description = "Folder id of account.", required = true)
    public String getFolderId() {
        return folderId;
    }

    @Schema(description = "Deleted flag.", required = true)
    public boolean isDeleted() {
        return deleted;
    }

    @Schema(description = "Accounts space id.")
    public String getAccountsSpacesId() {
        return accountsSpacesId;
    }

    @Schema(description = "Free tier flag.", required = true)
    @JsonProperty("freeTier")
    public boolean isFreeTier() {
        return freeTier;
    }

    @Schema(description = "Provider id of account.", required = true)
    public String getProviderId() {
        return providerId;
    }

    @Schema(description = "Reserve type.")
    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    public Optional<AccountReserveTypeDto> getReserveType() {
        return Optional.ofNullable(reserveType);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        AccountDto that = (AccountDto) o;
        return deleted == that.deleted &&
                freeTier == that.freeTier &&
                Objects.equals(id, that.id) &&
                Objects.equals(displayName, that.displayName) &&
                Objects.equals(folderId, that.folderId) &&
                Objects.equals(accountsSpacesId, that.accountsSpacesId) &&
                Objects.equals(providerId, that.providerId) &&
                Objects.equals(reserveType, that.reserveType);
    }

    @Override
    public int hashCode() {
        return Objects.hash(id, displayName, folderId, deleted, accountsSpacesId, freeTier, providerId, reserveType);
    }

    @Override
    public String toString() {
        return "AccountDto{" +
                "id='" + id + '\'' +
                ", displayName='" + displayName + '\'' +
                ", folderId='" + folderId + '\'' +
                ", deleted=" + deleted +
                ", accountsSpacesId='" + accountsSpacesId + '\'' +
                ", freeTier=" + freeTier +
                ", providerId='" + providerId + '\'' +
                ", reserveType=" + reserveType +
                '}';
    }
}
