package ru.yandex.intranet.d.web.model;

import java.util.List;
import java.util.Locale;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import io.swagger.v3.oas.annotations.media.Schema;

import ru.yandex.intranet.d.i18n.Locales;
import ru.yandex.intranet.d.model.resources.ResourceModel;

/**
 * Resources for front handles.
 *
 * @author Vladimir Zaytsev <vzay@yandex-team.ru>
 * @see ru.yandex.intranet.d.model.resources.ResourceModel
 * @see ru.yandex.intranet.d.web.model.resources.ResourceDto
 * @see ru.yandex.intranet.d.web.model.ResourceDto
 * @since 22-03-2021
 */
@Schema(description = "Resources for front handles.")
@JsonIgnoreProperties(ignoreUnknown = true)
public final class AvailableResourcesDto {
    private final List<Resource> resources;

    @JsonCreator
    public AvailableResourcesDto(List<Resource> resources) {
        this.resources = resources;
    }

    public AvailableResourcesDto(List<ResourceModel> models, Locale locale) {
        this.resources = models.stream().map(model -> new Resource(model, locale)).collect(Collectors.toList());
    }

    @Schema(description = "'Available resources list.", required = true)
    public List<Resource> getResources() {
        return resources;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        AvailableResourcesDto that = (AvailableResourcesDto) o;
        return Objects.equals(resources, that.resources);
    }

    @Override
    public int hashCode() {
        return Objects.hash(resources);
    }

    @Override
    public String toString() {
        return "AvailableResourcesDto{" +
                "resources=" + resources +
                '}';
    }

    @Schema(description = "Resource for front handles.")
    @JsonIgnoreProperties(ignoreUnknown = true)
    public static final class Resource {
        private final String id;
        private final String providerId;
        private final String accountsSpacesId;
        private final String key;
        private final String resourceTypeId;
        private final String name;
        private final String description;
        private final String unitsEnsembleId;
        private final boolean managed;
        private final boolean orderable;
        private final boolean readOnly;
        private final Set<ResourceSegmentSettings> segments;
        private final ResourceUnits resourceUnits;
        private final String baseUnitId;

        @JsonCreator
        @SuppressWarnings("ParameterNumber")
        public Resource(
                String id,
                String providerId,
                String accountsSpacesId,
                String key,
                String resourceTypeId,
                String name,
                String description,
                String unitsEnsembleId,
                boolean managed,
                boolean orderable,
                boolean readOnly,
                Set<ResourceSegmentSettings> segments,
                ResourceUnits resourceUnits,
                String baseUnitId
        ) {
            this.id = id;
            this.providerId = providerId;
            this.accountsSpacesId = accountsSpacesId;
            this.key = key;
            this.resourceTypeId = resourceTypeId;
            this.name = name;
            this.description = description;
            this.unitsEnsembleId = unitsEnsembleId;
            this.managed = managed;
            this.orderable = orderable;
            this.readOnly = readOnly;
            this.segments = segments;
            this.resourceUnits = resourceUnits;
            this.baseUnitId = baseUnitId;
        }

        public Resource(ResourceModel model, Locale locale) {
            this.id = model.getId();
            this.providerId = model.getProviderId();
            this.accountsSpacesId = model.getAccountsSpacesId();
            this.key = model.getKey();
            this.resourceTypeId = model.getResourceTypeId();
            this.name = Locales.select(model.getNameEn(), model.getNameRu(), locale);
            this.description = Locales.select(model.getDescriptionEn(), model.getDescriptionRu(), locale);
            this.unitsEnsembleId = model.getUnitsEnsembleId();
            this.managed = model.isManaged();
            this.orderable = model.isOrderable();
            this.readOnly = model.isReadOnly();
            this.segments = model.getSegments() != null
                    ? model.getSegments().stream().map(s -> new ResourceSegmentSettings(s.getSegmentationId(),
                            s.getSegmentId())).collect(Collectors.toSet())
                    : null;
            this.resourceUnits = model.getResourceUnits() != null
                    ? new ResourceUnits(model.getResourceUnits().getAllowedUnitIds(),
                            model.getResourceUnits().getDefaultUnitId())
                    : null;
            this.baseUnitId = model.getBaseUnitId();
        }

        @Schema(description = "Unique id.", required = true)
        public String getId() {
            return id;
        }

        @Schema(description = "Provider id.", required = true)
        public String getProviderId() {
            return providerId;
        }

        @Schema(description = "Accounts space id, if used")
        public String getAccountsSpacesId() {
            return accountsSpacesId;
        }

        @Schema(description = "Key, unique per provider (or per accounts space, if it used).", required = true)
        public String getKey() {
            return key;
        }

        @Schema(description = "Units ensemble id.", required = true)
        public String getUnitsEnsembleId() {
            return unitsEnsembleId;
        }

        @Schema(description = "Name.", required = true)
        public String getName() {
            return name;
        }

        @Schema(description = "Description.", required = true)
        public String getDescription() {
            return description;
        }

        @Schema(description = "'Read only' flag.", required = true)
        public boolean isReadOnly() {
            return readOnly;
        }

        @Schema(description = "'Resource can be ordered' flag.", required = true)
        public boolean isOrderable() {
            return orderable;
        }

        @Schema(description = "'Managed' flag.", required = true)
        public boolean isManaged() {
            return managed;
        }

        @Schema(description = "Resource type id.", required = true)
        public String getResourceTypeId() {
            return resourceTypeId;
        }

        @Schema(description = "Resource segments, if used.")
        @JsonInclude(JsonInclude.Include.NON_EMPTY)
        public Set<ResourceSegmentSettings> getSegments() {
            return segments;
        }

        @Schema(description = "Allowed and default for resource units.", required = true)
        public ResourceUnits getResourceUnits() {
            return resourceUnits;
        }

        @Schema(description = "Base unit for resource.", required = true)
        public String getBaseUnitId() {
            return baseUnitId;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            Resource resource = (Resource) o;
            return managed == resource.managed &&
                    orderable == resource.orderable &&
                    readOnly == resource.readOnly &&
                    Objects.equals(id, resource.id) &&
                    Objects.equals(providerId, resource.providerId) &&
                    Objects.equals(accountsSpacesId, resource.accountsSpacesId) &&
                    Objects.equals(key, resource.key) &&
                    Objects.equals(resourceTypeId, resource.resourceTypeId) &&
                    Objects.equals(name, resource.name) &&
                    Objects.equals(description, resource.description) &&
                    Objects.equals(unitsEnsembleId, resource.unitsEnsembleId) &&
                    Objects.equals(segments, resource.segments) &&
                    Objects.equals(resourceUnits, resource.resourceUnits) &&
                    Objects.equals(baseUnitId, resource.baseUnitId);
        }

        @Override
        public int hashCode() {
            return Objects.hash(id, providerId, accountsSpacesId, key, resourceTypeId, name, description,
                    unitsEnsembleId, managed, orderable, readOnly, segments, resourceUnits, baseUnitId);
        }

        @Override
        public String toString() {
            return "Resource{" +
                    "id='" + id + '\'' +
                    ", providerId='" + providerId + '\'' +
                    ", accountsSpacesId='" + accountsSpacesId + '\'' +
                    ", key='" + key + '\'' +
                    ", resourceTypeId='" + resourceTypeId + '\'' +
                    ", name='" + name + '\'' +
                    ", description='" + description + '\'' +
                    ", unitsEnsembleId='" + unitsEnsembleId + '\'' +
                    ", managed=" + managed +
                    ", orderable=" + orderable +
                    ", readOnly=" + readOnly +
                    ", segments=" + segments +
                    ", resourceUnits=" + resourceUnits +
                    ", baseUnitId='" + baseUnitId + '\'' +
                    '}';
        }
    }

    @Schema(description = "Resource segmentation and segment.")
    @JsonIgnoreProperties(ignoreUnknown = true)
    public static final class ResourceSegmentSettings {

        private final String segmentationId;
        private final String segmentId;

        @JsonCreator
        public ResourceSegmentSettings(String segmentationId, String segmentId) {
            this.segmentationId = segmentationId;
            this.segmentId = segmentId;
        }

        @Schema(description = "Resource segmentation id.", required = true)
        public String getSegmentationId() {
            return segmentationId;
        }

        @Schema(description = "Resource segment id.", required = true)
        public String getSegmentId() {
            return segmentId;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            ResourceSegmentSettings that = (ResourceSegmentSettings) o;
            return Objects.equals(segmentationId, that.segmentationId) &&
                    Objects.equals(segmentId, that.segmentId);
        }

        @Override
        public int hashCode() {
            return Objects.hash(segmentationId, segmentId);
        }

        @Override
        public String toString() {
            return "ResourceSegmentSettings{" +
                    "segmentationId='" + segmentationId + '\'' +
                    ", segmentId='" + segmentId + '\'' +
                    '}';
        }

    }

    @Schema(description = "Resource units.")
    @JsonIgnoreProperties(ignoreUnknown = true)
    public static final class ResourceUnits {

        private final Set<String> allowedUnitIds;
        private final String defaultUnitId;

        @JsonCreator
        public ResourceUnits(Set<String> allowedUnitIds, String defaultUnitId) {
            this.allowedUnitIds = allowedUnitIds;
            this.defaultUnitId = defaultUnitId;
        }

        @Schema(description = "Allowed unit ids.", required = true)
        public Set<String> getAllowedUnitIds() {
            return allowedUnitIds;
        }

        @Schema(description = "Default unit id.", required = true)
        public String getDefaultUnitId() {
            return defaultUnitId;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            ResourceUnits that = (ResourceUnits) o;
            return Objects.equals(allowedUnitIds, that.allowedUnitIds) &&
                    Objects.equals(defaultUnitId, that.defaultUnitId);
        }

        @Override
        public int hashCode() {
            return Objects.hash(allowedUnitIds, defaultUnitId);
        }

        @Override
        public String toString() {
            return "ResourceUnits{" +
                    "allowedUnitIds=" + allowedUnitIds +
                    ", defaultUnitId='" + defaultUnitId + '\'' +
                    '}';
        }

    }

}
