package ru.yandex.intranet.d.web.model;

import java.util.NoSuchElementException;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;

import javax.annotation.Nullable;

import org.apache.commons.lang3.builder.ReflectionToStringBuilder;

/**
 * Container for patched value in DTO
 *
 * @author Evgenii Serov <evserov@yandex-team.ru>
 */
public class PatchedValue<T> {

    private static final PatchedValue<?> NO_CHANGED = new PatchedValue<>();

    private final boolean isValueChanged;
    private final @Nullable T newValue;

    public PatchedValue(@Nullable T newValue) {
        this.newValue = newValue;
        this.isValueChanged = true;
    }

    private PatchedValue() {
        this.newValue = null;
        this.isValueChanged = false;
    }

    public T getNewValue() {
        if (isValueChanged()) {
            return newValue;
        } else {
            throw new NoSuchElementException("No new value present");
        }
    }

    public Optional<T> getNewValueInOptional() {
        return Optional.ofNullable(getNewValue());
    }

    public boolean isValueChanged() {
        return isValueChanged;
    }

    public void acceptNewValueToSetter(Consumer<T> setter, T defaultValue) {
        if (isValueChanged()) {
            setter.accept(getNewValue());
        } else {
            setter.accept(defaultValue);
        }
    }

    public void acceptNewValueOptionalToSetter(Consumer<Optional<T>> setter, T defaultValue) {
        if (isValueChanged()) {
            setter.accept(getNewValueInOptional());
        } else {
            setter.accept(Optional.ofNullable(defaultValue));
        }
    }

    public static <T> PatchedValue<T> noChanged() {
        @SuppressWarnings("unchecked")
        PatchedValue<T> t = (PatchedValue<T>) NO_CHANGED;
        return t;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        PatchedValue<?> that = (PatchedValue<?>) o;
        return isValueChanged == that.isValueChanged && Objects.equals(newValue, that.newValue);
    }

    @Override
    public int hashCode() {
        return Objects.hash(isValueChanged, newValue);
    }

    @Override
    public String toString() {
        return ReflectionToStringBuilder.toString(this);
    }
}
