package ru.yandex.intranet.d.web.model;

import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import javax.annotation.Nullable;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import io.swagger.v3.oas.annotations.media.Schema;
import org.apache.commons.lang3.builder.ReflectionToStringBuilder;

import ru.yandex.intranet.d.web.model.providers.ProviderUISettingsDto;
import ru.yandex.intranet.d.web.model.providers.RelatedResourceMappingDto;

/**
 * Provider DTO.
 *
 * @author Nikita Minin <spasitel@yandex-team.ru>
 */
@Schema(description = "Provider.")
@JsonIgnoreProperties(ignoreUnknown = true)
public final class ProviderDto {
    private final String id;
    private final String name;
    private final String description;
    private final boolean deleted;
    private final boolean readOnly;
    private final boolean managed;
    private final String key;
    private final String meteringKey;
    private final boolean multipleAccountsPerFolder;
    private final boolean accountTransferWithQuota;
    private final boolean importAllowed;
    private final boolean accountsSpacesSupported;
    private final boolean syncEnabled;
    private final AccountsSettingsDto accountsSettingsDto;
    private final Map<String, RelatedResourceMappingDto> relatedResourcesByResourceId;
    private final String reserveFolderId;
    private final boolean hasDefaultQuotas;
    private final FeatureStateDto allocatedSupported;
    private final @Nullable ProviderUISettingsDto uiSettings;

    @JsonCreator
    @SuppressWarnings("ParameterNumber")
    public ProviderDto(
            String id,
            String name,
            String description,
            boolean deleted,
            boolean readOnly,
            boolean managed,
            String key,
            String meteringKey,
            boolean multipleAccountsPerFolder,
            boolean accountTransferWithQuota,
            boolean importAllowed,
            boolean accountsSpacesSupported,
            boolean syncEnabled,
            AccountsSettingsDto accountsSettingsDto,
            Map<String, RelatedResourceMappingDto> relatedResourcesByResourceId,
            String reserveFolderId,
            boolean hasDefaultQuotas,
            FeatureStateDto allocatedSupported,
            @Nullable ProviderUISettingsDto uiSettings
    ) {
        this.id = id;
        this.name = name;
        this.description = description;
        this.deleted = deleted;
        this.readOnly = readOnly;
        this.managed = managed;
        this.key = key;
        this.meteringKey = meteringKey;
        this.multipleAccountsPerFolder = multipleAccountsPerFolder;
        this.accountTransferWithQuota = accountTransferWithQuota;
        this.importAllowed = importAllowed;
        this.accountsSpacesSupported = accountsSpacesSupported;
        this.syncEnabled = syncEnabled;
        this.accountsSettingsDto = accountsSettingsDto;
        this.relatedResourcesByResourceId = relatedResourcesByResourceId;
        this.reserveFolderId = reserveFolderId;
        this.hasDefaultQuotas = hasDefaultQuotas;
        this.allocatedSupported = allocatedSupported;
        this.uiSettings = uiSettings;
    }

    @Schema(description = "Provider id.", required = true)
    public String getId() {
        return id;
    }

    @Schema(description = "Provider name.", required = true)
    public String getName() {
        return name;
    }

    @Schema(description = "Description.", required = true)
    public String getDescription() {
        return description;
    }

    @Schema(description = "'Is deleted' flag.", required = true)
    public boolean isDeleted() {
        return deleted;
    }

    @Schema(description = "'Is read only' flag.", required = true)
    public boolean isReadOnly() {
        return readOnly;
    }

    @Schema(description = "'Is managed' flag.", required = true)
    public boolean isManaged() {
        return managed;
    }

    @Schema(description = "Key.", required = true)
    public String getKey() {
        return key;
    }

    @Schema(description = "Metering key.")
    public Optional<String> getMeteringKey() {
        return Optional.ofNullable(meteringKey);
    }

    @Schema(description = "Is this provider has multiple accounts per folder.", required = true)
    public boolean isMultipleAccountsPerFolder() {
        return multipleAccountsPerFolder;
    }

    @Schema(description = "Is this provider make transfer account with quota.", required = true)
    public boolean isAccountTransferWithQuota() {
        return accountTransferWithQuota;
    }

    @Schema(description = "Is this provider support import.", required = true)
    public boolean isImportAllowed() {
        return importAllowed;
    }

    @Schema(description = "Is this provider support accounts spaces.", required = true)
    public boolean isAccountsSpacesSupported() {
        return accountsSpacesSupported;
    }

    @Schema(description = "Is this provider has enabled sync.", required = true)
    public boolean isSyncEnabled() {
        return syncEnabled;
    }

    @Schema(description = "Provider account settings.", required = true)
    public AccountsSettingsDto getAccountsSettingsDto() {
        return accountsSettingsDto;
    }

    @Schema(description = "Resource mapping to relative resources.")
    public Optional<Map<String, RelatedResourceMappingDto>> getRelatedResourcesByResourceId() {
        return Optional.ofNullable(relatedResourcesByResourceId);
    }

    @Schema(description = "ID of provider's reserve folder, or null.")
    public String getReserveFolderId() {
        return reserveFolderId;
    }

    @Schema(description = "True, if provider has default quotas for some resources.", required = true)
    public boolean isHasDefaultQuotas() {
        return hasDefaultQuotas;
    }

    @Schema(description = "'Allocated supported' feature state", required = true)
    public FeatureStateDto isAllocatedSupported() {
        return allocatedSupported;
    }

    @Schema(description = "Providers UI settings.")
    public Optional<ProviderUISettingsDto> getUiSettings() {
        return Optional.ofNullable(uiSettings);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        ProviderDto that = (ProviderDto) o;
        return deleted == that.deleted &&
                readOnly == that.readOnly &&
                managed == that.managed &&
                multipleAccountsPerFolder == that.multipleAccountsPerFolder &&
                accountTransferWithQuota == that.accountTransferWithQuota &&
                importAllowed == that.importAllowed &&
                accountsSpacesSupported == that.accountsSpacesSupported &&
                syncEnabled == that.syncEnabled &&
                hasDefaultQuotas == that.hasDefaultQuotas &&
                Objects.equals(id, that.id) &&
                Objects.equals(name, that.name) &&
                Objects.equals(description, that.description) &&
                Objects.equals(key, that.key) &&
                Objects.equals(meteringKey, that.meteringKey) &&
                Objects.equals(accountsSettingsDto, that.accountsSettingsDto) &&
                Objects.equals(relatedResourcesByResourceId, that.relatedResourcesByResourceId) &&
                Objects.equals(reserveFolderId, that.reserveFolderId) &&
                Objects.equals(allocatedSupported, that.allocatedSupported) &&
                Objects.equals(uiSettings, that.uiSettings);
    }

    @Override
    public int hashCode() {
        return Objects.hash(id, name, description, deleted, readOnly, managed, key, meteringKey,
                multipleAccountsPerFolder, accountTransferWithQuota, importAllowed, accountsSpacesSupported,
                syncEnabled, accountsSettingsDto, relatedResourcesByResourceId, reserveFolderId, hasDefaultQuotas,
                allocatedSupported, uiSettings);
    }

    @Override
    public String toString() {
        return ReflectionToStringBuilder.toString(this);
    }

    @Schema(description = "Provider account settings")
    @JsonIgnoreProperties(ignoreUnknown = true)
    public static final class AccountsSettingsDto {
        private final boolean displayNameSupported;
        private final boolean keySupported;
        private final boolean deleteSupported;
        private final boolean softDeleteSupported;
        private final boolean moveSupported;
        private final boolean renameSupported;
        private final boolean perAccountVersionSupported;
        private final boolean perProvisionVersionSupported;
        private final boolean perAccountLastUpdateSupported;
        private final boolean perProvisionLastUpdateSupported;
        private final boolean operationIdDeduplicationSupported;
        private final boolean syncCoolDownDisabled;
        private final boolean retryCoolDownDisabled;
        private final long accountsSyncPageSize;
        private final boolean moveProvisionSupported;
        private final boolean multipleReservesAllowed;

        @JsonCreator
        @SuppressWarnings("ParameterNumber")
        public AccountsSettingsDto(
                boolean displayNameSupported,
                boolean keySupported,
                boolean deleteSupported,
                boolean softDeleteSupported,
                boolean moveSupported,
                boolean renameSupported,
                boolean perAccountVersionSupported,
                boolean perProvisionVersionSupported,
                boolean perAccountLastUpdateSupported,
                boolean perProvisionLastUpdateSupported,
                boolean operationIdDeduplicationSupported,
                boolean syncCoolDownDisabled,
                boolean retryCoolDownDisabled,
                long accountsSyncPageSize,
                boolean moveProvisionSupported,
                boolean multipleReservesAllowed) {
            this.displayNameSupported = displayNameSupported;
            this.keySupported = keySupported;
            this.deleteSupported = deleteSupported;
            this.softDeleteSupported = softDeleteSupported;
            this.moveSupported = moveSupported;
            this.renameSupported = renameSupported;
            this.perAccountVersionSupported = perAccountVersionSupported;
            this.perProvisionVersionSupported = perProvisionVersionSupported;
            this.perAccountLastUpdateSupported = perAccountLastUpdateSupported;
            this.perProvisionLastUpdateSupported = perProvisionLastUpdateSupported;
            this.operationIdDeduplicationSupported = operationIdDeduplicationSupported;
            this.syncCoolDownDisabled = syncCoolDownDisabled;
            this.retryCoolDownDisabled = retryCoolDownDisabled;
            this.accountsSyncPageSize = accountsSyncPageSize;
            this.moveProvisionSupported = moveProvisionSupported;
            this.multipleReservesAllowed = multipleReservesAllowed;
        }

        @Schema(description = "Is this provider support display name for accounts.", required = true)
        public boolean isDisplayNameSupported() {
            return displayNameSupported;
        }

        @Schema(description = "Is this provider support key for accounts.", required = true)
        public boolean isKeySupported() {
            return keySupported;
        }

        @Schema(description = "Is this provider support account deletion.", required = true)
        public boolean isDeleteSupported() {
            return deleteSupported;
        }

        @Schema(description = "Is this provider support account soft deletion.", required = true)
        public boolean isSoftDeleteSupported() {
            return softDeleteSupported;
        }

        @Schema(description = "Is this provider support account moving.", required = true)
        public boolean isMoveSupported() {
            return moveSupported;
        }

        @Schema(description = "Is this provider support account renaming.", required = true)
        public boolean isRenameSupported() {
            return renameSupported;
        }

        @Schema(description = "Is this provider support version per account.", required = true)
        public boolean isPerAccountVersionSupported() {
            return perAccountVersionSupported;
        }

        @Schema(description = "Is this provider support version per provision.", required = true)
        public boolean isPerProvisionVersionSupported() {
            return perProvisionVersionSupported;
        }

        @Schema(description = "Is this provider support last update for account.", required = true)
        public boolean isPerAccountLastUpdateSupported() {
            return perAccountLastUpdateSupported;
        }

        @Schema(description = "Is this provider support last update for provision.", required = true)
        public boolean isPerProvisionLastUpdateSupported() {
            return perProvisionLastUpdateSupported;
        }

        @Schema(description = "'Operation id deduplication is supported' flag", required = true)
        public boolean isOperationIdDeduplicationSupported() {
            return operationIdDeduplicationSupported;
        }

        @Schema(description = "'Synchronization cool down is disabled' flag", required = true)
        public boolean isSyncCoolDownDisabled() {
            return syncCoolDownDisabled;
        }

        @Schema(description = "'Retries cool down is disabled' flag", required = true)
        public boolean isRetryCoolDownDisabled() {
            return retryCoolDownDisabled;
        }

        @Schema(description = "Maximum page size of accounts sync response.", required = true)
        public long getAccountsSyncPageSize() {
            return accountsSyncPageSize;
        }

        @Schema(description = "'MoveProvision method is supported' flag", required = true)
        public boolean isMoveProvisionSupported() {
            return moveProvisionSupported;
        }

        @Schema(description = "'Multiple reserves allowed' flag", required = true)
        public boolean isMultipleReservesAllowed() {
            return multipleReservesAllowed;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            AccountsSettingsDto that = (AccountsSettingsDto) o;
            return displayNameSupported == that.displayNameSupported &&
                    keySupported == that.keySupported &&
                    deleteSupported == that.deleteSupported &&
                    softDeleteSupported == that.softDeleteSupported &&
                    moveSupported == that.moveSupported &&
                    renameSupported == that.renameSupported &&
                    perAccountVersionSupported == that.perAccountVersionSupported &&
                    perProvisionVersionSupported == that.perProvisionVersionSupported &&
                    perAccountLastUpdateSupported == that.perAccountLastUpdateSupported &&
                    perProvisionLastUpdateSupported == that.perProvisionLastUpdateSupported &&
                    operationIdDeduplicationSupported == that.operationIdDeduplicationSupported &&
                    syncCoolDownDisabled == that.syncCoolDownDisabled &&
                    retryCoolDownDisabled == that.retryCoolDownDisabled &&
                    accountsSyncPageSize == that.accountsSyncPageSize &&
                    moveProvisionSupported == that.moveProvisionSupported &&
                    multipleReservesAllowed == that.multipleReservesAllowed;
        }

        @Override
        public int hashCode() {
            return Objects.hash(displayNameSupported, keySupported, deleteSupported, softDeleteSupported,
                    moveSupported, renameSupported, perAccountVersionSupported, perProvisionVersionSupported,
                    perAccountLastUpdateSupported, perProvisionLastUpdateSupported, operationIdDeduplicationSupported,
                    syncCoolDownDisabled, retryCoolDownDisabled, accountsSyncPageSize, moveProvisionSupported,
                    multipleReservesAllowed);
        }

        @Override
        public String toString() {
            return ReflectionToStringBuilder.toString(this);
        }
    }
}
