package ru.yandex.intranet.d.web.model;

import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import io.swagger.v3.oas.annotations.media.Schema;

import ru.yandex.intranet.d.i18n.Locales;
import ru.yandex.intranet.d.model.resources.ResourceModel;
import ru.yandex.intranet.d.model.resources.segmentations.ResourceSegmentationModel;
import ru.yandex.intranet.d.model.resources.segments.ResourceSegmentModel;
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel;
import ru.yandex.intranet.d.services.units.UnitsComparator;
import ru.yandex.intranet.d.web.model.resources.ResourceSegmentationSegmentDto;
import ru.yandex.intranet.d.web.model.resources.ResourceUnitsDto;

import static ru.yandex.intranet.d.services.quotas.QuotasHelper.getUnitName;

/**
 * Resource DTO.
 *
 * @author Nikita Minin <spasitel@yandex-team.ru>
 */
@Schema(description = "Resource.")
@JsonIgnoreProperties(ignoreUnknown = true)
public final class ResourceDto {
    private final String id;
    private final String providerId;
    private final String specification;
    private final String displayName;
    private final String defaultUnit;
    private final String defaultUnitId;
    private final boolean deleted;
    private final String resourceTypeId;
    private final String unitsEnsembleId;
    private final String accountsSpacesId;
    private final boolean managed;
    private final boolean readOnly;
    private final boolean virtual;
    private final ResourceUnitsDto resourceUnits;
    private final Set<ResourceSegmentationSegmentDto> resourceSegments;
    private final FeatureStateDto allocatedSupported;

    @JsonCreator
    @SuppressWarnings("ParameterNumber")
    public ResourceDto(
            String id,
            String providerId,
            String specification,
            String displayName,
            String defaultUnit,
            String defaultUnitId,
            boolean deleted,
            String resourceTypeId,
            String unitsEnsembleId,
            String accountsSpacesId,
            boolean managed,
            boolean readOnly,
            boolean virtual,
            ResourceUnitsDto resourceUnits,
            Set<ResourceSegmentationSegmentDto> resourceSegments,
            FeatureStateDto allocatedSupported
    ) {
        this.id = id;
        this.providerId = providerId;
        this.specification = specification;
        this.displayName = displayName;
        this.defaultUnit = defaultUnit;
        this.defaultUnitId = defaultUnitId;
        this.deleted = deleted;
        this.resourceTypeId = resourceTypeId;
        this.unitsEnsembleId = unitsEnsembleId;
        this.accountsSpacesId = accountsSpacesId;
        this.managed = managed;
        this.readOnly = readOnly;
        this.virtual = virtual;
        this.resourceUnits = resourceUnits;
        this.resourceSegments = resourceSegments;
        this.allocatedSupported = allocatedSupported;
    }

    public ResourceDto(ResourceModel res, Locale locale) {
        this(res, locale, null, null, null);
    }

    public ResourceDto(
            ResourceModel res,
            Locale locale,
            @Nullable Map<String, UnitsEnsembleModel> unitsEnsemblesById,
            @Nullable Map<String, ResourceSegmentationModel> segmentationsById,
            @Nullable Map<String, ResourceSegmentModel> segmentsById
    ) {
        this.id = res.getId();
        this.providerId = res.getProviderId();
        this.specification = Locales.select(res.getDescriptionEn(), res.getDescriptionRu(), locale);
        this.displayName = Locales.select(res.getNameEn(), res.getNameRu(), locale);
        this.defaultUnit = unitsEnsemblesById == null ? null : unitsEnsemblesById.get(res.getUnitsEnsembleId())
                .unitById(res.getResourceUnits().getDefaultUnitId())
                .map(unitModel -> getUnitName(unitModel, locale))
                .orElseThrow(() -> new IllegalStateException(
                        "Unknown resource default unit: " + res.getResourceUnits().getDefaultUnitId()));
        this.defaultUnitId = res.getResourceUnits().getDefaultUnitId();
        this.deleted = res.isDeleted();
        this.resourceTypeId = res.getResourceTypeId();
        this.unitsEnsembleId = res.getUnitsEnsembleId();
        this.accountsSpacesId = res.getAccountsSpacesId();
        this.managed = res.isManaged();
        this.readOnly = res.isReadOnly();
        this.virtual = res.isVirtual();
        this.allocatedSupported = FeatureStateDto.fromModel(res.isAllocatedSupported().orElse(null));
        this.resourceUnits = new ResourceUnitsDto(
                unitsEnsemblesById == null ? List.of() : UnitsComparator.getSortedIds(
                        res.getResourceUnits().getAllowedUnitIds(),
                        unitsEnsemblesById.get(res.getUnitsEnsembleId())
                ),
                res.getResourceUnits().getDefaultUnitId(),
                res.getResourceUnits().getProviderApiUnitId());
        this.resourceSegments = res.getSegments() == null ? Set.of() : res.getSegments().stream().map(segment ->
                new ResourceSegmentationSegmentDto(segment, segmentationsById, segmentsById, locale)
        ).collect(Collectors.toSet());
    }

    @Schema(description = "Unique id of resource.", required = true)
    public String getId() {
        return id;
    }

    @Schema(description = "Provider id.", required = true)
    public String getProviderId() {
        return providerId;
    }

    @Schema(description = "Description of resource.", required = true)
    public String getSpecification() {
        return specification;
    }

    @Schema(description = "Display name of resource.", required = true)
    public String getDisplayName() {
        return displayName;
    }

    @Schema(description = "Default unit name.")
    public String getDefaultUnit() {
        return defaultUnit;
    }

    @Schema(description = "Default unit id.")
    public String getDefaultUnitId() {
        return defaultUnitId;
    }

    @Schema(description = "Deleted flag.", required = true)
    public boolean isDeleted() {
        return deleted;
    }

    @Schema(description = "Resource type id.", required = true)
    public String getResourceTypeId() {
        return resourceTypeId;
    }

    @Schema(description = "Units ensemble id.", required = true)
    public String getUnitsEnsembleId() {
        return unitsEnsembleId;
    }

    @Schema(description = "Accounts space id.")
    public String getAccountsSpacesId() {
        return accountsSpacesId;
    }

    @Schema(description = "'Managed' flag.", required = true)
    public boolean isManaged() {
        return managed;
    }

    @Schema(description = "'Read only' flag.", required = true)
    public boolean isReadOnly() {
        return readOnly;
    }

    @Schema(description = "'Virtual' flag.", required = true)
    public boolean isVirtual() {
        return virtual;
    }

    @Schema(description = "Allowed and default for resource units.", required = true)
    public ResourceUnitsDto getResourceUnits() {
        return resourceUnits;
    }

    @Schema(description = "Resource segments IDs by segmentations.", required = true)
    public Set<ResourceSegmentationSegmentDto> getResourceSegments() {
        return resourceSegments;
    }

    @Schema(description = "'Allocated supported' feature state.", required = true)
    public FeatureStateDto isAllocatedSupported() {
        return allocatedSupported;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        ResourceDto that = (ResourceDto) o;
        return deleted == that.deleted &&
                managed == that.managed &&
                readOnly == that.readOnly &&
                virtual == that.virtual &&
                Objects.equals(id, that.id) &&
                Objects.equals(providerId, that.providerId) &&
                Objects.equals(specification, that.specification) &&
                Objects.equals(displayName, that.displayName) &&
                Objects.equals(defaultUnit, that.defaultUnit) &&
                Objects.equals(defaultUnitId, that.defaultUnitId) &&
                Objects.equals(resourceTypeId, that.resourceTypeId) &&
                Objects.equals(unitsEnsembleId, that.unitsEnsembleId) &&
                Objects.equals(accountsSpacesId, that.accountsSpacesId) &&
                Objects.equals(resourceUnits, that.resourceUnits) &&
                Objects.equals(resourceSegments, that.resourceSegments) &&
                Objects.equals(allocatedSupported, that.allocatedSupported);
    }

    @Override
    public int hashCode() {
        return Objects.hash(
                id,
                providerId,
                specification,
                displayName,
                defaultUnit,
                defaultUnitId,
                deleted,
                resourceTypeId,
                unitsEnsembleId,
                accountsSpacesId,
                managed,
                readOnly,
                virtual,
                resourceUnits,
                resourceSegments,
                allocatedSupported
        );
    }

    @Override
    public String toString() {
        return "ResourceDto{" +
                "id='" + id + '\'' +
                ", providerId='" + providerId + '\'' +
                ", specification='" + specification + '\'' +
                ", displayName='" + displayName + '\'' +
                ", defaultUnit='" + defaultUnit + '\'' +
                ", defaultUnitId='" + defaultUnitId + '\'' +
                ", deleted=" + deleted +
                ", resourceTypeId='" + resourceTypeId + '\'' +
                ", unitsEnsembleId='" + unitsEnsembleId + '\'' +
                ", accountsSpacesId='" + accountsSpacesId + '\'' +
                ", managed=" + managed +
                ", readOnly=" + readOnly +
                ", virtual=" + virtual +
                ", resourceUnits=" + resourceUnits +
                ", resourceSegments=" + resourceSegments +
                ", allocatedSupported=" + allocatedSupported +
                '}';
    }
}
