package ru.yandex.intranet.d.web.model.accounts;

import java.util.Objects;
import java.util.Optional;

import javax.annotation.Nullable;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import io.swagger.v3.oas.annotations.media.Schema;

/**
 * Account DTO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Schema(description = "Account.")
@JsonIgnoreProperties(ignoreUnknown = true)
public final class AccountDto {

    private final String id;
    private final String externalId;
    private final String externalKey;
    private final long version;
    private final String folderId;
    private final String providerId;
    private final String displayName;
    private final String accountsSpaceId;
    private final boolean freeTier;
    @Nullable
    private final AccountReserveTypeDto reserveType;

    @JsonCreator
    @SuppressWarnings("ParameterNumber")
    public AccountDto(String id,
                      String externalId,
                      String externalKey,
                      long version,
                      String folderId,
                      String providerId,
                      String displayName,
                      String accountsSpaceId,
                      boolean freeTier,
                      @Nullable AccountReserveTypeDto reserveType) {
        this.id = id;
        this.externalId = externalId;
        this.externalKey = externalKey;
        this.version = version;
        this.folderId = folderId;
        this.providerId = providerId;
        this.displayName = displayName;
        this.accountsSpaceId = accountsSpaceId;
        this.freeTier = freeTier;
        this.reserveType = reserveType;
    }

    public static Builder builder() {
        return new Builder();
    }

    @Schema(description = "Unique id.", required = true)
    public String getId() {
        return id;
    }

    @Schema(description = "Unique id in provider.", required = true)
    public String getExternalId() {
        return externalId;
    }

    @Schema(description = "Unique key in provider.")
    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    public Optional<String> getExternalKey() {
        return Optional.ofNullable(externalKey);
    }

    @Schema(description = "Version.", required = true)
    public long getVersion() {
        return version;
    }

    @Schema(description = "Folder id.", required = true)
    public String getFolderId() {
        return folderId;
    }

    @Schema(description = "Provider id.", required = true)
    public String getProviderId() {
        return providerId;
    }

    @Schema(description = "Display name.")
    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    public Optional<String> getDisplayName() {
        return Optional.ofNullable(displayName);
    }

    @Schema(description = "Accounts space id.")
    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    public Optional<String> getAccountsSpaceId() {
        return Optional.ofNullable(accountsSpaceId);
    }

    @Schema(description = "'Is free tier account' flag.", required = true)
    public boolean isFreeTier() {
        return freeTier;
    }

    @Schema(description = "Reserve type.")
    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    public Optional<AccountReserveTypeDto> getReserveType() {
        return Optional.ofNullable(reserveType);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        AccountDto that = (AccountDto) o;
        return version == that.version &&
                freeTier == that.freeTier &&
                Objects.equals(id, that.id) &&
                Objects.equals(externalId, that.externalId) &&
                Objects.equals(externalKey, that.externalKey) &&
                Objects.equals(folderId, that.folderId) &&
                Objects.equals(providerId, that.providerId) &&
                Objects.equals(displayName, that.displayName) &&
                Objects.equals(accountsSpaceId, that.accountsSpaceId) &&
                Objects.equals(reserveType, that.reserveType);
    }

    @Override
    public int hashCode() {
        return Objects.hash(id, externalId, externalKey, version, folderId, providerId, displayName,
                accountsSpaceId, freeTier, reserveType);
    }

    @Override
    public String toString() {
        return "AccountDto{" +
                "id='" + id + '\'' +
                ", externalId='" + externalId + '\'' +
                ", externalKey='" + externalKey + '\'' +
                ", version=" + version +
                ", folderId='" + folderId + '\'' +
                ", providerId='" + providerId + '\'' +
                ", displayName='" + displayName + '\'' +
                ", accountsSpaceId='" + accountsSpaceId + '\'' +
                ", freeTier=" + freeTier +
                ", reserveType=" + reserveType +
                '}';
    }

    public static final class Builder {

        private String id;
        private String externalId;
        private String externalKey;
        private Long version;
        private String folderId;
        private String providerId;
        private String displayName;
        private String accountsSpaceId;
        private Boolean freeTier;
        private AccountReserveTypeDto reserveType;

        private Builder() {
        }

        public Builder id(String id) {
            this.id = id;
            return this;
        }

        public Builder externalId(String externalId) {
            this.externalId = externalId;
            return this;
        }

        public Builder externalKey(String externalKey) {
            this.externalKey = externalKey;
            return this;
        }

        public Builder version(long version) {
            this.version = version;
            return this;
        }

        public Builder folderId(String folderId) {
            this.folderId = folderId;
            return this;
        }

        public Builder providerId(String providerId) {
            this.providerId = providerId;
            return this;
        }

        public Builder displayName(String displayName) {
            this.displayName = displayName;
            return this;
        }

        public Builder accountsSpaceId(String accountsSpaceId) {
            this.accountsSpaceId = accountsSpaceId;
            return this;
        }

        public Builder freeTier(boolean freeTier) {
            this.freeTier = freeTier;
            return this;
        }

        public Builder reserveType(AccountReserveTypeDto reserveType) {
            this.reserveType = reserveType;
            return this;
        }

        public AccountDto build() {
            Objects.requireNonNull(id, "Id is required");
            Objects.requireNonNull(externalId, "External id is required");
            Objects.requireNonNull(version, "Version is required");
            Objects.requireNonNull(folderId, "Folder id is required");
            Objects.requireNonNull(providerId, "Provider id is required");
            Objects.requireNonNull(freeTier, "Free tier is required");
            return new AccountDto(id, externalId, externalKey, version, folderId, providerId, displayName,
                    accountsSpaceId, freeTier, reserveType);
        }

    }

}
