package ru.yandex.intranet.d.web.model.accounts;

import java.util.Objects;
import java.util.Optional;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import io.swagger.v3.oas.annotations.media.Schema;

/**
 * Account operation DTO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Schema(description = "Account operation result.")
@JsonIgnoreProperties(ignoreUnknown = true)
public final class AccountOperationDto {

    private final AccountOperationStatusDto operationStatus;
    private final AccountDto result;
    private final String operationId;

    @JsonCreator
    public AccountOperationDto(AccountOperationStatusDto operationStatus,
                               AccountDto result,
                               String operationId) {
        this.operationStatus = operationStatus;
        this.result = result;
        this.operationId = operationId;
    }

    public static Builder builder() {
        return new Builder();
    }

    @Schema(description = "Operation status.", required = true)
    public AccountOperationStatusDto getOperationStatus() {
        return operationStatus;
    }

    @Schema(description = "Account.")
    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    public Optional<AccountDto> getResult() {
        return Optional.ofNullable(result);
    }

    @Schema(description = "Unique operation id.", required = true)
    public String getOperationId() {
        return operationId;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        AccountOperationDto that = (AccountOperationDto) o;
        return operationStatus == that.operationStatus &&
                Objects.equals(result, that.result) &&
                Objects.equals(operationId, that.operationId);
    }

    @Override
    public int hashCode() {
        return Objects.hash(operationStatus, result, operationId);
    }

    @Override
    public String toString() {
        return "AccountOperationDto{" +
                "operationStatus=" + operationStatus +
                ", result=" + result +
                ", operationId='" + operationId + '\'' +
                '}';
    }

    public static final class Builder {

        private AccountOperationStatusDto operationStatus;
        private AccountDto result;
        private String operationId;

        private Builder() {
        }

        public Builder operationStatus(AccountOperationStatusDto operationStatus) {
            this.operationStatus = operationStatus;
            return this;
        }

        public Builder result(AccountDto result) {
            this.result = result;
            return this;
        }

        public Builder operationId(String operationId) {
            this.operationId = operationId;
            return this;
        }

        public AccountOperationDto build() {
            Objects.requireNonNull(operationStatus, "Operation status is required");
            Objects.requireNonNull(operationId, "Operation id is required");
            return new AccountOperationDto(operationStatus, result, operationId);
        }

    }

}
