package ru.yandex.intranet.d.web.model.accounts;

import java.util.List;
import java.util.Objects;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import io.swagger.v3.oas.annotations.media.Schema;

import ru.yandex.intranet.d.services.integration.providers.rest.model.AccountsSpaceKeyResponseDto;
import ru.yandex.intranet.d.web.model.folders.front.ExpandedAccountResource;

/**
 * Account with quota.
 *
 * @author Ruslan Kadriev <aqru@yandex-team.ru>
 */
@Schema(description = "Account with quota.")
@JsonIgnoreProperties(ignoreUnknown = true)
public class AccountWithQuotaDto {
    private final String id;
    private final String displayName;
    private final String accountsSpaceId;
    private final AccountsSpaceKeyResponseDto accountsSpaceKey;
    private final List<ExpandedAccountResource> provisions;

    @JsonCreator
    public AccountWithQuotaDto(String id, String displayName, String accountsSpaceId,
                               AccountsSpaceKeyResponseDto accountsSpaceKey, List<ExpandedAccountResource> provisions) {
        this.id = id;
        this.displayName = displayName;
        this.accountsSpaceId = accountsSpaceId;
        this.accountsSpaceKey = accountsSpaceKey;
        this.provisions = provisions;
    }

    @Schema(description = "Unique id.", required = true)
    public String getId() {
        return id;
    }

    @Schema(description = "Display name.")
    public String getDisplayName() {
        return displayName;
    }

    @Schema(description = "Accounts space id.")
    public String getAccountsSpaceId() {
        return accountsSpaceId;
    }

    @Schema(description = "Accounts space key.")
    public AccountsSpaceKeyResponseDto getAccountsSpaceKey() {
        return accountsSpaceKey;
    }

    @Schema(description = "Account provisions.")
    public List<ExpandedAccountResource> getProvisions() {
        return provisions;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        AccountWithQuotaDto that = (AccountWithQuotaDto) o;
        return Objects.equals(id, that.id) && Objects.equals(displayName, that.displayName)
                && Objects.equals(accountsSpaceId, that.accountsSpaceId)
                && Objects.equals(accountsSpaceKey, that.accountsSpaceKey)
                && Objects.equals(provisions, that.provisions);
    }

    @Override
    public int hashCode() {
        return Objects.hash(id, displayName, accountsSpaceId, accountsSpaceKey, provisions);
    }

    @Override
    public String toString() {
        return "AccountWithQuotaDto{" +
                "id='" + id + '\'' +
                ", displayName='" + displayName + '\'' +
                ", accountsSpaceId='" + accountsSpaceId + '\'' +
                ", accountsSpaceKey=" + accountsSpaceKey +
                ", provisions=" + provisions +
                '}';
    }

    public static Builder builder() {
        return new Builder();
    }

    public static class Builder {
        private String id;
        private String displayName;
        private String accountsSpaceId;
        private AccountsSpaceKeyResponseDto accountsSpaceKey;
        private List<ExpandedAccountResource> provisions;

        public Builder id(String id) {
            this.id = id;
            return this;
        }

        public Builder displayName(String displayName) {
            this.displayName = displayName;
            return this;
        }

        public Builder accountsSpaceId(String accountsSpaceId) {
            this.accountsSpaceId = accountsSpaceId;
            return this;
        }

        public Builder accountsSpaceKey(AccountsSpaceKeyResponseDto accountsSpaceKey) {
            this.accountsSpaceKey = accountsSpaceKey;
            return this;
        }

        public Builder provisions(List<ExpandedAccountResource> provisions) {
            this.provisions = provisions;
            return this;
        }

        public AccountWithQuotaDto build() {
            return new AccountWithQuotaDto(
                    Objects.requireNonNull(id, "Id is required"),
                    displayName,
                    accountsSpaceId,
                    accountsSpaceKey,
                    provisions
            );
        }
    }
}
