package ru.yandex.intranet.d.web.model.accounts;

import java.util.Objects;
import java.util.Optional;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import io.swagger.v3.oas.annotations.media.Schema;

/**
 * Create account DTO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Schema(description = "Account.")
@JsonIgnoreProperties(ignoreUnknown = true)
public final class CreateAccountDto {

    private final String providerId;
    private final String accountsSpaceId;
    private final String externalKey;
    private final String displayName;
    private final Boolean freeTier;
    private final AccountReserveTypeInputDto reserveType;

    @JsonCreator
    public CreateAccountDto(String providerId,
                            String accountsSpaceId,
                            String externalKey,
                            String displayName,
                            Boolean freeTier,
                            AccountReserveTypeInputDto reserveType) {
        this.providerId = providerId;
        this.accountsSpaceId = accountsSpaceId;
        this.externalKey = externalKey;
        this.displayName = displayName;
        this.freeTier = freeTier;
        this.reserveType = reserveType;
    }

    public static Builder builder() {
        return new Builder();
    }

    @Schema(description = "Provider id.", required = true)
    public Optional<String> getProviderId() {
        return Optional.ofNullable(providerId);
    }

    @Schema(description = "Accounts space id.")
    public Optional<String> getAccountsSpaceId() {
        return Optional.ofNullable(accountsSpaceId);
    }

    @Schema(description = "Unique key in provider.")
    public Optional<String> getExternalKey() {
        return Optional.ofNullable(externalKey);
    }

    @Schema(description = "Display name.")
    public Optional<String> getDisplayName() {
        return Optional.ofNullable(displayName);
    }

    @Schema(description = "'Is free tier account' flag, default value is false.")
    public Optional<Boolean> getFreeTier() {
        return Optional.ofNullable(freeTier);
    }

    @Schema(description = "Reserve type.")
    public Optional<AccountReserveTypeInputDto> getReserveType() {
        return Optional.ofNullable(reserveType);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        CreateAccountDto that = (CreateAccountDto) o;
        return Objects.equals(providerId, that.providerId) &&
                Objects.equals(accountsSpaceId, that.accountsSpaceId) &&
                Objects.equals(externalKey, that.externalKey) &&
                Objects.equals(displayName, that.displayName) &&
                Objects.equals(freeTier, that.freeTier) &&
                Objects.equals(reserveType, that.reserveType);
    }

    @Override
    public int hashCode() {
        return Objects.hash(providerId, accountsSpaceId, externalKey, displayName, freeTier, reserveType);
    }

    @Override
    public String toString() {
        return "CreateAccountDto{" +
                "providerId='" + providerId + '\'' +
                ", accountsSpaceId='" + accountsSpaceId + '\'' +
                ", externalKey='" + externalKey + '\'' +
                ", displayName='" + displayName + '\'' +
                ", freeTier=" + freeTier +
                ", reserveType=" + reserveType +
                '}';
    }

    public static final class Builder {

        private String providerId;
        private String accountsSpaceId;
        private String externalKey;
        private String displayName;
        private Boolean freeTier;
        private AccountReserveTypeInputDto reserveType;

        private Builder() {
        }

        public Builder providerId(String providerId) {
            this.providerId = providerId;
            return this;
        }

        public Builder accountsSpaceId(String accountsSpaceId) {
            this.accountsSpaceId = accountsSpaceId;
            return this;
        }

        public Builder externalKey(String externalKey) {
            this.externalKey = externalKey;
            return this;
        }

        public Builder displayName(String displayName) {
            this.displayName = displayName;
            return this;
        }

        public Builder freeTier(Boolean freeTier) {
            this.freeTier = freeTier;
            return this;
        }

        public Builder reserveType(AccountReserveTypeInputDto reserveType) {
            this.reserveType = reserveType;
            return this;
        }

        public CreateAccountDto build() {
            return new CreateAccountDto(providerId, accountsSpaceId, externalKey, displayName, freeTier, reserveType);
        }

    }

}
