package ru.yandex.intranet.d.web.model.coordination;

import java.util.Objects;
import java.util.Optional;
import java.util.Set;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import io.swagger.v3.oas.annotations.media.Schema;

/**
 * Cluster state DTO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Schema(description = "Cluster state.")
@JsonIgnoreProperties(ignoreUnknown = true)
public final class ClusterStateDto {

    private final boolean leader;
    private final boolean follower;
    private final boolean member;
    private final ClusterNodeDto leaderNode;
    private final Set<ClusterNodeDto> memberNodes;

    @JsonCreator
    public ClusterStateDto(@JsonProperty("leader") boolean leader,
                           @JsonProperty("follower") boolean follower,
                           @JsonProperty("member") boolean member,
                           @JsonProperty("leaderNode") ClusterNodeDto leaderNode,
                           @JsonProperty("memberNodes") Set<ClusterNodeDto> memberNodes) {
        this.leader = leader;
        this.follower = follower;
        this.member = member;
        this.leaderNode = leaderNode;
        this.memberNodes = memberNodes;
    }

    @Schema(description = "Is this node the leader.", required = true)
    public boolean isLeader() {
        return leader;
    }

    @Schema(description = "Is this node a follower.", required = true)
    public boolean isFollower() {
        return follower;
    }

    @Schema(description = "Is this node a member.", required = true)
    public boolean isMember() {
        return member;
    }

    @Schema(description = "Leader node info.", required = true)
    public Optional<ClusterNodeDto> getLeaderNode() {
        return Optional.ofNullable(leaderNode);
    }

    @Schema(description = "Member nodes info.", required = true)
    public Set<ClusterNodeDto> getMemberNodes() {
        return memberNodes;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        ClusterStateDto that = (ClusterStateDto) o;
        return leader == that.leader &&
                follower == that.follower &&
                member == that.member &&
                Objects.equals(leaderNode, that.leaderNode) &&
                Objects.equals(memberNodes, that.memberNodes);
    }

    @Override
    public int hashCode() {
        return Objects.hash(leader, follower, member, leaderNode, memberNodes);
    }

    @Override
    public String toString() {
        return "ClusterStateDto{" +
                "leader=" + leader +
                ", follower=" + follower +
                ", member=" + member +
                ", leaderNode=" + leaderNode +
                ", memberNodes=" + memberNodes +
                '}';
    }

}
