package ru.yandex.intranet.d.web.model.delivery;

import java.time.Instant;
import java.util.Objects;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonFormat;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.google.common.base.Preconditions;
import io.swagger.v3.oas.annotations.media.Schema;

/**
 * Deliverable folder operation DTO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Schema(description = "Deliverable folder operation.")
@JsonIgnoreProperties(ignoreUnknown = true)
public final class DeliverableFolderOperationDto {

    private final String id;
    private final String folderId;
    private final Instant timestamp;

    @JsonCreator
    public DeliverableFolderOperationDto(
            String id,
            String folderId,
            @JsonFormat(shape = JsonFormat.Shape.STRING, pattern = "yyyy-MM-dd'T'HH:mm:ss.SSSSSS'Z'", timezone = "UTC")
            Instant timestamp) {
        this.id = id;
        this.folderId = folderId;
        this.timestamp = timestamp;
    }

    public static Builder builder() {
        return new Builder();
    }

    @Schema(description = "Operation id.", required = true)
    public String getId() {
        return id;
    }

    @Schema(description = "Folder id.", required = true)
    public String getFolderId() {
        return folderId;
    }

    @Schema(description = "Operation timestamp.", required = true)
    @JsonFormat(shape = JsonFormat.Shape.STRING, pattern = "yyyy-MM-dd'T'HH:mm:ss.SSSSSS'Z'", timezone = "UTC")
    public Instant getTimestamp() {
        return timestamp;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        DeliverableFolderOperationDto that = (DeliverableFolderOperationDto) o;
        return Objects.equals(id, that.id) &&
                Objects.equals(folderId, that.folderId) &&
                Objects.equals(timestamp, that.timestamp);
    }

    @Override
    public int hashCode() {
        return Objects.hash(id, folderId, timestamp);
    }

    @Override
    public String toString() {
        return "DeliverableFolderOperationDto{" +
                "id='" + id + '\'' +
                ", folderId='" + folderId + '\'' +
                ", timestamp=" + timestamp +
                '}';
    }

    public static final class Builder {

        private String id;
        private String folderId;
        private Instant timestamp;

        private Builder() {
        }

        public Builder id(String id) {
            this.id = id;
            return this;
        }

        public Builder folderId(String folderId) {
            this.folderId = folderId;
            return this;
        }

        public Builder timestamp(Instant timestamp) {
            this.timestamp = timestamp;
            return this;
        }

        public DeliverableFolderOperationDto build() {
            Preconditions.checkNotNull(id, "Id is required");
            Preconditions.checkNotNull(folderId, "FolderId is required");
            Preconditions.checkNotNull(timestamp, "Timestamp is required");
            return new DeliverableFolderOperationDto(id, folderId, timestamp);
        }

    }

}
