package ru.yandex.intranet.d.web.model.delivery;

import java.util.Objects;
import java.util.Optional;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import io.swagger.v3.oas.annotations.media.Schema;

/**
 * Deliverable request DTO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Schema(description = "Deliverable request.")
@JsonIgnoreProperties(ignoreUnknown = true)
public final class DeliverableRequestDto {

    private final Long serviceId;
    private final String folderId;
    private final String resourceId;
    private final DeliverableDeltaDto delta;
    private final DeliverableMetaRequestDto meta;

    @JsonCreator
    public DeliverableRequestDto(Long serviceId, String folderId, String resourceId, DeliverableDeltaDto delta,
                                 DeliverableMetaRequestDto meta) {
        this.serviceId = serviceId;
        this.folderId = folderId;
        this.resourceId = resourceId;
        this.delta = delta;
        this.meta = meta;
    }

    public static Builder builder() {
        return new Builder();
    }

    @Schema(description = "Service id.")
    public Optional<Long> getServiceId() {
        return Optional.ofNullable(serviceId);
    }

    @Schema(description = "Folder id.")
    public Optional<String> getFolderId() {
        return Optional.ofNullable(folderId);
    }

    @Schema(description = "Resource id.", required = true)
    public Optional<String> getResourceId() {
        return Optional.ofNullable(resourceId);
    }

    @Schema(description = "Delta.", required = true)
    public Optional<DeliverableDeltaDto> getDelta() {
        return Optional.ofNullable(delta);
    }

    @Schema(description = "Metadata.", required = true)
    public Optional<DeliverableMetaRequestDto> getMeta() {
        return Optional.ofNullable(meta);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        DeliverableRequestDto that = (DeliverableRequestDto) o;
        return Objects.equals(serviceId, that.serviceId) &&
                Objects.equals(folderId, that.folderId) &&
                Objects.equals(resourceId, that.resourceId) &&
                Objects.equals(delta, that.delta) &&
                Objects.equals(meta, that.meta);
    }

    @Override
    public int hashCode() {
        return Objects.hash(serviceId, folderId, resourceId, delta, meta);
    }

    @Override
    public String toString() {
        return "DeliverableRequestDto{" +
                "serviceId=" + serviceId +
                ", folderId='" + folderId + '\'' +
                ", resourceId='" + resourceId + '\'' +
                ", delta=" + delta +
                ", meta=" + meta +
                '}';
    }

    public static final class Builder {

        private Long serviceId;
        private String folderId;
        private String resourceId;
        private DeliverableDeltaDto delta;
        private DeliverableMetaRequestDto meta;

        private Builder() {
        }

        public Builder serviceId(long serviceId) {
            this.serviceId = serviceId;
            return this;
        }

        public Builder folderId(String folderId) {
            this.folderId = folderId;
            return this;
        }

        public Builder resourceId(String resourceId) {
            this.resourceId = resourceId;
            return this;
        }

        public Builder delta(DeliverableDeltaDto delta) {
            this.delta = delta;
            return this;
        }

        public Builder meta(DeliverableMetaRequestDto meta) {
            this.meta = meta;
            return this;
        }

        public DeliverableRequestDto build() {
            return new DeliverableRequestDto(serviceId, folderId, resourceId, delta, meta);
        }

    }

}
