package ru.yandex.intranet.d.web.model.delivery;

import java.util.Objects;
import java.util.Optional;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.google.common.base.Preconditions;
import io.swagger.v3.oas.annotations.media.Schema;

/**
 * Deliverable response DTO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Schema(description = "Deliverable request.")
@JsonIgnoreProperties(ignoreUnknown = true)
public final class DeliverableResponseDto {

    private final Long serviceId;
    private final String folderId;
    private final String resourceId;
    private final DeliverableFolderOperationDto folderOperationLog;
    private final DeliverableMetaResponseDto meta;

    @JsonCreator
    public DeliverableResponseDto(Long serviceId, String folderId, String resourceId,
                                  DeliverableFolderOperationDto folderOperationLog, DeliverableMetaResponseDto meta) {
        this.serviceId = serviceId;
        this.folderId = folderId;
        this.resourceId = resourceId;
        this.folderOperationLog = folderOperationLog;
        this.meta = meta;
    }

    public static Builder builder() {
        return new Builder();
    }

    @Schema(description = "Service id.")
    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    public Optional<Long> getServiceId() {
        return Optional.ofNullable(serviceId);
    }

    @Schema(description = "Folder id.")
    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    public Optional<String> getFolderId() {
        return Optional.ofNullable(folderId);
    }

    @Schema(description = "Resource id.", required = true)
    public String getResourceId() {
        return resourceId;
    }

    @Schema(description = "Folder operation log id.", required = true)
    public DeliverableFolderOperationDto getFolderOperationLog() {
        return folderOperationLog;
    }

    @Schema(description = "Metadata.", required = true)
    public DeliverableMetaResponseDto getMeta() {
        return meta;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        DeliverableResponseDto that = (DeliverableResponseDto) o;
        return Objects.equals(serviceId, that.serviceId) &&
                Objects.equals(folderId, that.folderId) &&
                Objects.equals(resourceId, that.resourceId) &&
                Objects.equals(folderOperationLog, that.folderOperationLog) &&
                Objects.equals(meta, that.meta);
    }

    @Override
    public int hashCode() {
        return Objects.hash(serviceId, folderId, resourceId, folderOperationLog, meta);
    }

    @Override
    public String toString() {
        return "DeliverableResponseDto{" +
                "serviceId=" + serviceId +
                ", folderId='" + folderId + '\'' +
                ", resourceId='" + resourceId + '\'' +
                ", folderOperationLog=" + folderOperationLog +
                ", meta=" + meta +
                '}';
    }

    public static final class Builder {

        private Long serviceId;
        private String folderId;
        private String resourceId;
        private DeliverableFolderOperationDto folderOperationLog;
        private DeliverableMetaResponseDto meta;

        private Builder() {
        }

        public Builder serviceId(long serviceId) {
            this.serviceId = serviceId;
            return this;
        }

        public Builder folderId(String folderId) {
            this.folderId = folderId;
            return this;
        }

        public Builder resourceId(String resourceId) {
            this.resourceId = resourceId;
            return this;
        }

        public Builder folderOperationLog(DeliverableFolderOperationDto folderOperationLog) {
            this.folderOperationLog = folderOperationLog;
            return this;
        }

        public Builder meta(DeliverableMetaResponseDto meta) {
            this.meta = meta;
            return this;
        }

        public DeliverableResponseDto build() {
            Preconditions.checkNotNull(resourceId, "ResourceId is required");
            Preconditions.checkNotNull(folderOperationLog, "FolderOperationLog is required");
            Preconditions.checkNotNull(meta, "Meta is required");
            Preconditions.checkArgument(serviceId != null || folderId != null,
                    "Either ServiceId or FolderId is required");
            return new DeliverableResponseDto(serviceId, folderId, resourceId, folderOperationLog, meta);
        }

    }

}
