package ru.yandex.intranet.d.web.model.delivery;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import io.swagger.v3.oas.annotations.media.Schema;

/**
 * Delivery request DTO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Schema(description = "Delivery request.")
@JsonIgnoreProperties(ignoreUnknown = true)
public final class DeliveryRequestDto {

    private final String deliveryId;
    private final String authorUid;
    private final List<DeliverableRequestDto> deliverables;

    @JsonCreator
    public DeliveryRequestDto(String deliveryId, String authorUid, List<DeliverableRequestDto> deliverables) {
        this.deliveryId = deliveryId;
        this.authorUid = authorUid;
        this.deliverables = deliverables;
    }

    public static Builder builder() {
        return new Builder();
    }

    @Schema(description = "Unique id.", required = true)
    public Optional<String> getDeliveryId() {
        return Optional.ofNullable(deliveryId);
    }

    @Schema(description = "User uid.", required = true)
    public Optional<String> getAuthorUid() {
        return Optional.ofNullable(authorUid);
    }

    @Schema(description = "Deliveries.", required = true)
    public Optional<List<DeliverableRequestDto>> getDeliverables() {
        return Optional.ofNullable(deliverables);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        DeliveryRequestDto that = (DeliveryRequestDto) o;
        return Objects.equals(deliveryId, that.deliveryId) &&
                Objects.equals(authorUid, that.authorUid) &&
                Objects.equals(deliverables, that.deliverables);
    }

    @Override
    public int hashCode() {
        return Objects.hash(deliveryId, authorUid, deliverables);
    }

    @Override
    public String toString() {
        return "DeliveryRequestDto{" +
                "deliveryId='" + deliveryId + '\'' +
                ", authorUid='" + authorUid + '\'' +
                ", deliverables=" + deliverables +
                '}';
    }

    public static final class Builder {

        private final List<DeliverableRequestDto> deliverables = new ArrayList<>();

        private String deliveryId;
        private String authorUid;

        private Builder() {
        }

        public Builder deliveryId(String deliveryId) {
            this.deliveryId = deliveryId;
            return this;
        }

        public Builder authorUid(String authorUid) {
            this.authorUid = authorUid;
            return this;
        }

        public Builder addDeliverable(DeliverableRequestDto deliverable) {
            this.deliverables.add(deliverable);
            return this;
        }

        public DeliveryRequestDto build() {
            return new DeliveryRequestDto(deliveryId, authorUid, deliverables);
        }

    }

}
