package ru.yandex.intranet.d.web.model.folders;

import java.util.Objects;
import java.util.Set;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import io.swagger.v3.oas.annotations.media.Schema;

import ru.yandex.intranet.d.model.folders.FolderModel;
import ru.yandex.intranet.d.model.folders.FolderType;

/**
 * Frontend folder DTO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Schema(description = "Folder.")
@JsonIgnoreProperties(ignoreUnknown = true)
public final class FrontFolderDto {

    private final String id;
    private final long version;
    private final long serviceId;
    private final FolderType folderType;
    private final String displayName;
    private final String description;
    private final boolean deleted;
    private final Set<String> tags;

    @SuppressWarnings("ParameterNumber")
    @JsonCreator
    public FrontFolderDto(
            @JsonProperty("id") String id,
            @JsonProperty("version")long version,
            @JsonProperty("serviceId") long serviceId,
            @JsonProperty("folderType") FolderType folderType,
            @JsonProperty("displayName") String displayName,
            @JsonProperty("description") String description,
            @JsonProperty("deleted") boolean deleted,
            @JsonProperty("tags") Set<String> tags
    ) {
        this.id = id;
        this.version = version;
        this.serviceId = serviceId;
        this.folderType = folderType;
        this.displayName = displayName;
        this.description = description;
        this.deleted = deleted;
        this.tags = tags;
    }

    public static FrontFolderDto from(FolderModel folder) {
        return new FrontFolderDto(
                folder.getId(),
                folder.getVersion(),
                folder.getServiceId(),
                folder.getFolderType(),
                folder.getDisplayName(),
                folder.getDescription().orElse(null),
                folder.isDeleted(),
                folder.getTags()
        );
    }

    @Schema(description = "Id.", required = true)
    public String getId() {
        return id;
    }

    @Schema(description = "Service id.", required = true)
    public long getServiceId() {
        return serviceId;
    }

    @Schema(description = "Display name.", required = true)
    public String getDisplayName() {
        return displayName;
    }

    @Schema(description = "'Is deleted' flag.", required = true)
    public boolean isDeleted() {
        return deleted;
    }

    @Schema(description = "Version.", required = true)
    public long getVersion() {
        return version;
    }

    @Schema(description = "Folder type.", required = true)
    public FolderType getFolderType() {
        return folderType;
    }

    @Schema(description = "Description.")
    public String getDescription() {
        return description;
    }

    @Schema(description = "Tags.", required = true)
    public Set<String> getTags() {
        return tags;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        FrontFolderDto folderDto = (FrontFolderDto) o;
        return version == folderDto.version &&
                serviceId == folderDto.serviceId &&
                deleted == folderDto.deleted &&
                Objects.equals(id, folderDto.id) &&
                folderType == folderDto.folderType &&
                Objects.equals(displayName, folderDto.displayName) &&
                Objects.equals(description, folderDto.description) &&
                Objects.equals(tags, folderDto.tags);
    }

    @Override
    public int hashCode() {
        return Objects.hash(id, version, serviceId, folderType, displayName, description, deleted, tags);
    }

    @Override
    public String toString() {
        return "FrontFolderDto{" +
                "id='" + id + '\'' +
                ", version=" + version +
                ", serviceId=" + serviceId +
                ", folderType=" + folderType +
                ", displayName='" + displayName + '\'' +
                ", description='" + description + '\'' +
                ", deleted=" + deleted +
                ", tags=" + tags +
                '}';
    }
}
