package ru.yandex.intranet.d.web.model.folders.front;

import java.math.BigDecimal;
import java.util.Objects;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import io.swagger.v3.oas.annotations.media.Schema;

import ru.yandex.intranet.d.web.model.AmountDto;

/**
 * Quotas in account on single resource.
 * Like ExpandedResource, but without unusable fields.
 *
 * @author Vladimir Zaytsev <vzay@yandex-team.ru>
 * @see ExpandedResource
 * @since 11.02.2021
 */
@Schema(description = "Account resource quota.")
@JsonIgnoreProperties(ignoreUnknown = true)
public final class ExpandedAccountResource {
    private final String resourceId;
    private final AmountDto provided;
    private final BigDecimal providedRatio;
    private final AmountDto allocated;
    private final BigDecimal allocatedRatio;
    private final AmountDto providedAndNotAllocated;

    @JsonCreator
    public ExpandedAccountResource(
            String resourceId,
            AmountDto provided,
            BigDecimal providedRatio,
            AmountDto allocated,
            BigDecimal allocatedRatio,
            AmountDto providedAndNotAllocated
    ) {
        this.resourceId = resourceId;
        this.provided = provided;
        this.providedRatio = providedRatio;
        this.allocated = allocated;
        this.allocatedRatio = allocatedRatio;
        this.providedAndNotAllocated = providedAndNotAllocated;
    }

    @Schema(description = "Resource id.", required = true)
    public String getResourceId() {
        return resourceId;
    }

    @Schema(description = "Provided to account quota.", required = true)
    public AmountDto getProvided() {
        return provided;
    }

    @Schema(description = "Provided quota divide total quota.", required = true)
    public BigDecimal getProvidedRatio() {
        return providedRatio;
    }

    @Schema(description = "Allocated in account quota.", required = true)
    public AmountDto getAllocated() {
        return allocated;
    }

    @Schema(description = "Allocated quota divide provided quota.", required = true)
    public BigDecimal getAllocatedRatio() {
        return allocatedRatio;
    }

    @Schema(description = "Provided and not allocated to account quota.", required = true)
    public AmountDto getProvidedAndNotAllocated() {
        return providedAndNotAllocated;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        ExpandedAccountResource that = (ExpandedAccountResource) o;
        return Objects.equals(resourceId, that.resourceId) && Objects.equals(provided, that.provided)
                && Objects.equals(providedRatio, that.providedRatio) && Objects.equals(allocated, that.allocated)
                && Objects.equals(allocatedRatio, that.allocatedRatio)
                && Objects.equals(providedAndNotAllocated, that.providedAndNotAllocated);
    }

    @Override
    public int hashCode() {
        return Objects.hash(resourceId, provided, providedRatio, allocated, allocatedRatio, providedAndNotAllocated);
    }

    @Override
    public String toString() {
        return "ExpandedAccountResource{" +
                "resourceId='" + resourceId + '\'' +
                ", provided=" + provided +
                ", providedRatio=" + providedRatio +
                ", allocated=" + allocated +
                ", allocatedRatio=" + allocatedRatio +
                ", providedAndNotAllocated=" + providedAndNotAllocated +
                '}';
    }
}
