package ru.yandex.intranet.d.web.model.folders.front.history;

import java.time.Instant;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

import javax.annotation.Nullable;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonFormat;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import io.swagger.v3.oas.annotations.media.Schema;
import org.apache.commons.lang3.builder.ReflectionToStringBuilder;

import ru.yandex.intranet.d.model.folders.FolderOperationType;
import ru.yandex.intranet.d.model.folders.FolderType;
import ru.yandex.intranet.d.model.folders.OperationPhase;
import ru.yandex.intranet.d.model.folders.TransferMetaHistoryModel.RoleInTransfer;
import ru.yandex.intranet.d.web.model.AmountDto;
import ru.yandex.intranet.d.web.model.accounts.AccountReserveTypeDto;

/**
 * Front folder operation log DTO
 *
 * @author Denis Blokhin <denblo@yandex-team.ru>
 */
@Schema(description = "Row for folder operation log.")
@JsonIgnoreProperties(ignoreUnknown = true)
@JsonInclude(JsonInclude.Include.NON_NULL)
public final class FrontFolderOperationLogDto {

    private final String id;
    private final String folderId;
    private final Instant dateTime;
    private final FolderOperationType operationType;
    @Nullable
    private final String authorUserId;
    @Nullable
    private final String authorProviderId;
    @Nullable
    private final String sourceFolderOperationsLogId;
    @Nullable
    private final String destinationFolderOperationsLogId;
    @Nullable
    private final FoldersFields oldFolderFields;
    @Nullable
    private final FoldersFields newFolderFields;
    private final AmountByResourceId oldBalance;
    private final AmountByResourceId newBalance;
    private final AmountByResourceId balanceDelta;
    private final ProvisionsByAccountId oldProvisions;
    private final ProvisionsByAccountId newProvisions;
    private final ProvisionDeltasByAccountId provisionsDelta;
    private final AmountByResourceId oldQuotas;
    private final AmountByResourceId newQuotas;
    private final AmountByResourceId quotasDelta;
    @Nullable
    private final String accountsQuotasOperationsId;
    @Nullable
    private final Accounts oldAccounts;
    @Nullable
    private final Accounts newAccounts;
    @Nullable
    private final String providerRequestId;
    @Nullable
    private final String quotasDemandsId;
    @Nullable
    private final OperationPhase operationPhase;
    private final Long order;
    @Nullable
    private final DeliverableMetaHistoryDto deliveryMeta;
    @Nullable
    private final TransferMetaHistoryDto transferMeta;

    @SuppressWarnings("checkstyle:ParameterNumber")
    @JsonCreator
    public FrontFolderOperationLogDto(
            String id,
            String folderId,
            Instant dateTime,
            FolderOperationType operationType,
            @Nullable String authorUserId,
            @Nullable String authorProviderId,
            @Nullable String sourceFolderOperationsLogId,
            @Nullable String destinationFolderOperationsLogId,
            @Nullable FoldersFields oldFolderFields,
            @Nullable FoldersFields newFolderFields,
            AmountByResourceId oldBalance,
            AmountByResourceId newBalance,
            AmountByResourceId balanceDelta,
            ProvisionsByAccountId oldProvisions,
            ProvisionsByAccountId newProvisions,
            ProvisionDeltasByAccountId provisionsDelta,
            AmountByResourceId oldQuotas,
            AmountByResourceId newQuotas,
            AmountByResourceId quotasDelta,
            @Nullable String accountsQuotasOperationsId,
            @Nullable Accounts oldAccounts,
            @Nullable Accounts newAccounts,
            @Nullable String providerRequestId,
            @Nullable String quotasDemandsId,
            @Nullable OperationPhase operationPhase,
            Long order,
            @Nullable DeliverableMetaHistoryDto deliveryMeta,
            @Nullable TransferMetaHistoryDto transferMeta
    ) {
        this.id = id;
        this.folderId = folderId;
        this.dateTime = dateTime;
        this.operationType = operationType;
        this.authorUserId = authorUserId;
        this.authorProviderId = authorProviderId;
        this.sourceFolderOperationsLogId = sourceFolderOperationsLogId;
        this.destinationFolderOperationsLogId = destinationFolderOperationsLogId;
        this.oldFolderFields = oldFolderFields;
        this.newFolderFields = newFolderFields;
        this.oldBalance = oldBalance;
        this.newBalance = newBalance;
        this.balanceDelta = balanceDelta;
        this.oldProvisions = oldProvisions;
        this.newProvisions = newProvisions;
        this.provisionsDelta = provisionsDelta;
        this.oldQuotas = oldQuotas;
        this.newQuotas = newQuotas;
        this.quotasDelta = quotasDelta;
        this.accountsQuotasOperationsId = accountsQuotasOperationsId;
        this.oldAccounts = oldAccounts;
        this.newAccounts = newAccounts;
        this.providerRequestId = providerRequestId;
        this.quotasDemandsId = quotasDemandsId;
        this.operationPhase = operationPhase;
        this.order = order;
        this.deliveryMeta = deliveryMeta;
        this.transferMeta = transferMeta;
    }

    @Schema(description = "Unique id of log row.", required = true)
    public String getId() {
        return id;
    }

    @Schema(description = "Folder id.", required = true)
    public String getFolderId() {
        return folderId;
    }

    @JsonFormat(shape = JsonFormat.Shape.STRING, pattern = "yyyy-MM-dd'T'HH:mm:ss.SSS'Z'", timezone = "UTC")
    @Schema(description = "Operation timestamp.", required = true)
    public Instant getDateTime() {
        return dateTime;
    }

    @Schema(description = """
            Type of operation with folder.
            <li> FOLDER_CREATE -- Folder created. Field 'newFolderFields' filled.
            <li> FOLDER_EDIT -- Folder fields changed, for example, folder renamed. Fields 'oldFolderFields'
             and 'newFolderFields' filled.
            <li> FOLDER_DELETE -- Folder deleted. Field 'oldFolderFields' filled.
            <li> PROVIDE_REVOKE_QUOTAS_TO_ACCOUNT -- Provision changed. Fields 'oldQuotas', 'newQuotas',
             'accountsQuotasOperationsId' and 'operationPhase' filled.
            <li> QUOTA_IMPORT -- Import folders quotas by provider admin.
            <li> SYNC_WITH_PROVIDER -- Periodical synchronization with provider changes accounts quotas.
            <li> QUOTA_TRANSFER -- Folder quotas changed when a transfer request has been applied.
            <li> RESERVE_TRANSFER -- Folder quotas changed when was applied a request from provider reserve.
            <li> CREATE_ACCOUNT -- Account created.
            <li> BULK_QUOTA_MOVE -- Administrator move quotas.
            <li> QUOTA_DELIVERY -- Hardware order was delivered.
            <li> CLEAN_QUOTAS_BY_ADMIN -- Administrator remove an outdated resource.
            <li> PROVISION_AS_QUOTA_BY_ADMIN -- Administrator copied accounts quotas to folder quotas.
            <li> PROVISION_TRANSFER -- Accounts quotas changed when a provision transfer request has been applied.
            """,
            required = true)
    public FolderOperationType getOperationType() {
        return operationType;
    }

    @Schema(description = "Author of operation, if it was a user.")
    @Nullable
    public String getAuthorUserId() {
        return authorUserId;
    }

    @Schema(description = "Author of operation, if it was a provider.")
    @Nullable
    public String getAuthorProviderId() {
        return authorProviderId;
    }

    @Schema(description = "Source folder, if this folder was a destination in operation.")
    @Nullable
    public String getSourceFolderOperationsLogId() {
        return sourceFolderOperationsLogId;
    }

    @Schema(description = "Destination folder, if this folder was a source in operation.")
    @Nullable
    public String getDestinationFolderOperationsLogId() {
        return destinationFolderOperationsLogId;
    }

    @Schema(description = "Old values of folder fields, if it changed.")
    @Nullable
    public FoldersFields getOldFolderFields() {
        return oldFolderFields;
    }

    @Schema(description = "New values of folder fields, if it changed.")
    @Nullable
    public FoldersFields getNewFolderFields() {
        return newFolderFields;
    }

    @Schema(description = "Old value of balance.", required = true)
    public AmountByResourceId getOldBalance() {
        return oldBalance;
    }

    @Schema(description = "New value of balance.", required = true)
    public AmountByResourceId getNewBalance() {
        return newBalance;
    }

    @Schema(description = "Delta balance.", required = true)
    public AmountByResourceId getBalanceDelta() {
        return balanceDelta;
    }

    @Schema(description = "Old provisions.", required = true)
    public ProvisionsByAccountId getOldProvisions() {
        return oldProvisions;
    }

    @Schema(description = "New provisions.", required = true)
    public ProvisionsByAccountId getNewProvisions() {
        return newProvisions;
    }

    @Schema(description = "Delta provisions.", required = true)
    public ProvisionDeltasByAccountId getProvisionsDelta() {
        return provisionsDelta;
    }

    @Schema(description = "Old quotas.", required = true)
    public AmountByResourceId getOldQuotas() {
        return oldQuotas;
    }

    @Schema(description = "New quotas.", required = true)
    public AmountByResourceId getNewQuotas() {
        return newQuotas;
    }

    @Schema(description = "Delta quotas.", required = true)
    public AmountByResourceId getQuotasDelta() {
        return quotasDelta;
    }

    @Schema(description = "Id of accounts quotas operation, if accounts quotas changed.")
    @Nullable
    public String getAccountsQuotasOperationsId() {
        return accountsQuotasOperationsId;
    }

    @Schema(description = "Old accounts, if it changed.")
    @Nullable
    public Accounts getOldAccounts() {
        return oldAccounts;
    }

    @Schema(description = "New accounts, if it changed.")
    @Nullable
    public Accounts getNewAccounts() {
        return newAccounts;
    }

    @Schema(description = "Id of request to provider, if present.")
    @Nullable
    public String getProviderRequestId() {
        return providerRequestId;
    }

    @Schema(description = "Id of quotas demand, if present.")
    @Nullable
    public String getQuotasDemandsId() {
        return quotasDemandsId;
    }

    @Nullable
    @Schema(description = "Operation phase, if present.")
    public OperationPhase getOperationPhase() {
        return operationPhase;
    }

    @Schema(description = "Order or log row.", required = true)
    public Long getOrder() {
        return order;
    }

    @Schema(description = "Delivery metadata, if present.")
    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    public Optional<DeliverableMetaHistoryDto> getDeliveryMeta() {
        return Optional.ofNullable(deliveryMeta);
    }

    @Schema(description = "Transfer metadata, if present.")
    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    public Optional<TransferMetaHistoryDto> getTransferMeta() {
        return Optional.ofNullable(transferMeta);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        FrontFolderOperationLogDto that = (FrontFolderOperationLogDto) o;
        return Objects.equals(id, that.id) && Objects.equals(folderId, that.folderId) && Objects.equals(dateTime,
                that.dateTime) && operationType == that.operationType && Objects.equals(authorUserId,
                that.authorUserId) && Objects.equals(authorProviderId, that.authorProviderId)
                && Objects.equals(sourceFolderOperationsLogId, that.sourceFolderOperationsLogId)
                && Objects.equals(destinationFolderOperationsLogId, that.destinationFolderOperationsLogId)
                && Objects.equals(oldFolderFields, that.oldFolderFields)
                && Objects.equals(newFolderFields, that.newFolderFields) && Objects.equals(oldBalance, that.oldBalance)
                && Objects.equals(newBalance, that.newBalance) && Objects.equals(balanceDelta, that.balanceDelta)
                && Objects.equals(oldProvisions, that.oldProvisions)
                && Objects.equals(newProvisions, that.newProvisions)
                && Objects.equals(provisionsDelta, that.provisionsDelta) && Objects.equals(oldQuotas, that.oldQuotas)
                && Objects.equals(newQuotas, that.newQuotas) && Objects.equals(quotasDelta, that.quotasDelta)
                && Objects.equals(accountsQuotasOperationsId, that.accountsQuotasOperationsId)
                && Objects.equals(oldAccounts, that.oldAccounts) && Objects.equals(newAccounts, that.newAccounts)
                && Objects.equals(providerRequestId, that.providerRequestId)
                && Objects.equals(quotasDemandsId, that.quotasDemandsId) && operationPhase == that.operationPhase
                && Objects.equals(order, that.order) && Objects.equals(deliveryMeta, that.deliveryMeta)
                && Objects.equals(transferMeta, that.transferMeta);
    }

    @Override
    public int hashCode() {
        return Objects.hash(id, folderId, dateTime, operationType, authorUserId, authorProviderId,
                sourceFolderOperationsLogId, destinationFolderOperationsLogId, oldFolderFields, newFolderFields,
                oldBalance, newBalance, balanceDelta, oldProvisions, newProvisions, provisionsDelta, oldQuotas,
                newQuotas, quotasDelta, accountsQuotasOperationsId, oldAccounts, newAccounts, providerRequestId,
                quotasDemandsId, operationPhase, order, deliveryMeta, transferMeta);
    }

    @Override
    public String toString() {
        return ReflectionToStringBuilder.toString(this);
    }

    @Schema(description = "Folder fields.")
    @JsonIgnoreProperties(ignoreUnknown = true)
    @JsonInclude(JsonInclude.Include.NON_NULL)
    public static class FoldersFields {
        private final Long serviceId;
        private final Long version;
        private final String displayName;
        private final String description;
        private final FolderType folderType;
        private final Set<String> tags;

        @JsonCreator
        public FoldersFields(@Nullable @JsonProperty("serviceId") Long serviceId,
                             @Nullable @JsonProperty("version") Long version,
                             @Nullable @JsonProperty("displayName") String displayName,
                             @Nullable @JsonProperty("description") String description,
                             @Nullable @JsonProperty("folderType") FolderType folderType,
                             @Nullable @JsonProperty("tags") Set<String> tags) {
            this.serviceId = serviceId;
            this.version = version;
            this.displayName = displayName;
            this.description = description;
            this.folderType = folderType;
            this.tags = tags;
        }

        @Nullable
        @Schema(description = "Service id.")
        public Long getServiceId() {
            return serviceId;
        }

        @Nullable
        @Schema(description = "Version.")
        public Long getVersion() {
            return version;
        }

        @Nullable
        @Schema(description = "Display name.")
        public String getDisplayName() {
            return displayName;
        }

        @Nullable
        @Schema(description = "Description.")
        public String getDescription() {
            return description;
        }

        @Nullable
        @Schema(description = "Folder type.")
        public FolderType getFolderType() {
            return folderType;
        }

        @Nullable
        @Schema(description = "Tags.")
        public Set<String> getTags() {
            return tags;
        }
    }

    @Schema(description = "Accounts.")
    @JsonIgnoreProperties(ignoreUnknown = true)
    @JsonInclude(JsonInclude.Include.NON_NULL)
    public static class Accounts {
        private final Map<String, Account> accounts;

        @JsonCreator
        public Accounts(@JsonProperty("accounts") Map<String, Account> accounts) {
            this.accounts = accounts;
        }

        @Schema(description = "Accounts by id.", required = true)
        public Map<String, Account> getAccounts() {
            return accounts;
        }
    }

    @Schema(description = "Account.")
    @JsonInclude(JsonInclude.Include.NON_NULL)
    @JsonIgnoreProperties(ignoreUnknown = true)
    public static class Account {
        private final Long version;
        private final String providerId;
        private final String outerAccountIdInProvider;
        private final String outerAccountKeyInProvider;
        private final String folderId;
        private final String displayName;
        private final Long lastReceivedVersion;
        private final String accountsSpaceId;
        @Nullable
        private final AccountReserveTypeDto reserveType;

        @SuppressWarnings("checkstyle:ParameterNumber")
        public Account(Long version,
                       String providerId,
                       String outerAccountIdInProvider,
                       String outerAccountKeyInProvider,
                       String folderId,
                       String displayName,
                       Long lastReceivedVersion,
                       String accountsSpaceId,
                       @Nullable
                       AccountReserveTypeDto reserveType) {
            this.version = version;
            this.providerId = providerId;
            this.outerAccountIdInProvider = outerAccountIdInProvider;
            this.outerAccountKeyInProvider = outerAccountKeyInProvider;
            this.folderId = folderId;
            this.displayName = displayName;
            this.lastReceivedVersion = lastReceivedVersion;
            this.accountsSpaceId = accountsSpaceId;
            this.reserveType = reserveType;
        }

        @Schema(description = "Version.")
        public Long getVersion() {
            return version;
        }

        @Schema(description = "Provider id.")
        public String getProviderId() {
            return providerId;
        }

        @Schema(description = "External id.")
        public String getOuterAccountIdInProvider() {
            return outerAccountIdInProvider;
        }

        @Schema(description = "External key.")
        public String getOuterAccountKeyInProvider() {
            return outerAccountKeyInProvider;
        }

        @Schema(description = "Folder id.")
        public String getFolderId() {
            return folderId;
        }

        @Schema(description = "Display name.")
        public String getDisplayName() {
            return displayName;
        }

        @Schema(description = "Last received version.")
        public Long getLastReceivedVersion() {
            return lastReceivedVersion;
        }

        @Schema(description = "Accounts spaces id")
        public String getAccountsSpaceId() {
            return accountsSpaceId;
        }

        @Schema(description = "Reserve type")
        @JsonInclude(JsonInclude.Include.NON_EMPTY)
        public Optional<AccountReserveTypeDto> getReserveType() {
            return Optional.ofNullable(reserveType);
        }
    }

    @Schema(description = "Amount by resource id.")
    @JsonIgnoreProperties(ignoreUnknown = true)
    @JsonInclude(JsonInclude.Include.NON_NULL)
    public static final class AmountByResourceId {
        private final Map<String, AmountDto> amountByResourceId;

        @JsonCreator
        public AmountByResourceId(@JsonProperty("amountByResourceId") Map<String, AmountDto> amountByResourceId) {
            this.amountByResourceId = amountByResourceId;
        }

        @Schema(description = "Amount by resource id.", required = true)
        public Map<String, AmountDto> getAmountByResourceId() {
            return amountByResourceId;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            AmountByResourceId that = (AmountByResourceId) o;
            return Objects.equals(amountByResourceId, that.amountByResourceId);
        }

        @Override
        public int hashCode() {
            return Objects.hash(amountByResourceId);
        }

        @Override
        public String toString() {
            return "AmountByResourceId{" +
                    "amountByResourceId=" + amountByResourceId +
                    '}';
        }
    }

    @Schema(description = "Provision by account id.")
    @JsonIgnoreProperties(ignoreUnknown = true)
    @JsonInclude(JsonInclude.Include.NON_NULL)
    public static class ProvisionsByAccountId {
        private final Map<String, ProvisionsByResourceId> provisionByByAccountId;

        @JsonCreator
        public ProvisionsByAccountId(
                @JsonProperty("provisionByByAccountId") Map<String, ProvisionsByResourceId> provisionByByAccountId) {
            this.provisionByByAccountId = provisionByByAccountId;
        }

        @Schema(description = "Provision by account id.", required = true)
        public Map<String, ProvisionsByResourceId> getProvisionByByAccountId() {
            return provisionByByAccountId;
        }
    }

    @Schema(description = "Provisions by resource id.")
    @JsonIgnoreProperties(ignoreUnknown = true)
    @JsonInclude(JsonInclude.Include.NON_NULL)
    public static class ProvisionsByResourceId {
        private final Map<String, Provision> provisionByResourceId;

        @JsonCreator
        public ProvisionsByResourceId(
                @JsonProperty("provisionByResourceId") Map<String, Provision> provisionByResourceId) {
            this.provisionByResourceId = provisionByResourceId;
        }

        @Schema(description = "Provision by resource id.", required = true)
        public Map<String, Provision> getProvisionByResourceId() {
            return provisionByResourceId;
        }
    }

    @Schema(description = "Provision.")
    @JsonIgnoreProperties(ignoreUnknown = true)
    @JsonInclude(JsonInclude.Include.NON_NULL)
    public static class Provision {
        private final AmountDto provision;
        private final Long version;

        @JsonCreator
        public Provision(@JsonProperty("provision") AmountDto provision, @JsonProperty("version") Long version) {
            this.provision = provision;
            this.version = version;
        }

        @Schema(description = "Provided amount.", required = true)
        public AmountDto getProvision() {
            return provision;
        }

        @Schema(description = "Version.")
        public Long getVersion() {
            return version;
        }
    }

    @Schema(description = "Provision delta by account id.")
    @JsonIgnoreProperties(ignoreUnknown = true)
    @JsonInclude(JsonInclude.Include.NON_NULL)
    public static final class ProvisionDeltasByAccountId {
        private final Map<String, ProvisionDeltasByResourceId> provisionDeltasByAccountId;

        @JsonCreator
        public ProvisionDeltasByAccountId(
                Map<String, ProvisionDeltasByResourceId> provisionDeltasByAccountId) {
            this.provisionDeltasByAccountId = provisionDeltasByAccountId;
        }

        @Schema(description = "Provision deltas by account id.", required = true)
        public Map<String, ProvisionDeltasByResourceId> getProvisionDeltasByAccountId() {
            return provisionDeltasByAccountId;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            ProvisionDeltasByAccountId that = (ProvisionDeltasByAccountId) o;
            return Objects.equals(provisionDeltasByAccountId, that.provisionDeltasByAccountId);
        }

        @Override
        public int hashCode() {
            return Objects.hash(provisionDeltasByAccountId);
        }

        @Override
        public String toString() {
            return "ProvisionDeltasByAccountId{" +
                    "provisionDeltasByAccountId=" + provisionDeltasByAccountId +
                    '}';
        }
    }

    @Schema(description = "Provision deltas by resource id.")
    @JsonIgnoreProperties(ignoreUnknown = true)
    @JsonInclude(JsonInclude.Include.NON_NULL)
    public static final class ProvisionDeltasByResourceId {
        private final Map<String, AmountDto> provisionDeltaByResourceId;

        @JsonCreator
        public ProvisionDeltasByResourceId(Map<String, AmountDto> provisionDeltaByResourceId) {
            this.provisionDeltaByResourceId = provisionDeltaByResourceId;
        }

        @Schema(description = "Provision delta by resource id.", required = true)
        public Map<String, AmountDto> getProvisionDeltaByResourceId() {
            return provisionDeltaByResourceId;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            ProvisionDeltasByResourceId that = (ProvisionDeltasByResourceId) o;
            return Objects.equals(provisionDeltaByResourceId, that.provisionDeltaByResourceId);
        }

        @Override
        public int hashCode() {
            return Objects.hash(provisionDeltaByResourceId);
        }

        @Override
        public String toString() {
            return "ProvisionDeltasByResourceId{" +
                    "provisionDeltaByResourceId=" + provisionDeltaByResourceId +
                    '}';
        }
    }

    @Schema(description = "Quota delivery metadata.")
    @JsonIgnoreProperties(ignoreUnknown = true)
    public static final class DeliverableMetaHistoryDto {

        private final long quotaRequestId;
        private final long campaignId;
        private final Set<Long> bigOrderIds;
        private final String deliveryId;

        @JsonCreator
        public DeliverableMetaHistoryDto(@JsonProperty("quotaRequestId") long quotaRequestId,
                                         @JsonProperty("campaignId") long campaignId,
                                         @JsonProperty("bigOrderIds") Set<Long> bigOrderIds,
                                         @JsonProperty("deliveryId") String deliveryId) {
            this.quotaRequestId = quotaRequestId;
            this.campaignId = campaignId;
            this.bigOrderIds = bigOrderIds;
            this.deliveryId = deliveryId;
        }

        @Schema(description = "Quota request id.", required = true)
        public long getQuotaRequestId() {
            return quotaRequestId;
        }

        @Schema(description = "Campaign id.", required = true)
        public long getCampaignId() {
            return campaignId;
        }

        @Schema(description = "Big order ids.", required = true)
        public Set<Long> getBigOrderIds() {
            return bigOrderIds;
        }

        @Schema(description = "Delivery id.", required = true)
        public String getDeliveryId() {
            return deliveryId;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            DeliverableMetaHistoryDto that = (DeliverableMetaHistoryDto) o;
            return quotaRequestId == that.quotaRequestId && campaignId == that.campaignId
                    && Objects.equals(bigOrderIds, that.bigOrderIds) && Objects.equals(deliveryId, that.deliveryId);
        }

        @Override
        public int hashCode() {
            return Objects.hash(quotaRequestId, campaignId, bigOrderIds, deliveryId);
        }

        @Override
        public String toString() {
            return "DeliverableMetaHistoryDto{" +
                    "quotaRequestId=" + quotaRequestId +
                    ", campaignId=" + campaignId +
                    ", bigOrderIds=" + bigOrderIds +
                    ", deliveryId='" + deliveryId + '\'' +
                    '}';
        }
    }

    @Schema(description = "Transfer request metadata.")
    @JsonIgnoreProperties(ignoreUnknown = true)
    public static final class TransferMetaHistoryDto {
        private final String transferRequestId;
        private final RoleInTransfer roleInTransfer;
        private final Set<AnotherDto> anotherParticipants;

        @JsonCreator
        public TransferMetaHistoryDto(
                String transferRequestId,
                RoleInTransfer roleInTransfer,
                Set<AnotherDto> anotherParticipants
        ) {
            this.transferRequestId = transferRequestId;
            this.roleInTransfer = roleInTransfer;
            this.anotherParticipants = anotherParticipants;
        }

        @Schema(description = "Transfer request ID.", required = true)
        public String getTransferRequestId() {
            return transferRequestId;
        }

        @Schema(description = "Role of this folder in transfer request (may be inapplicable).", required = true)
        public RoleInTransfer getRoleInTransfer() {
            return roleInTransfer;
        }

        @Schema(description = "Another transfer request participants IDs.", required = true)
        public Set<AnotherDto> getAnotherParticipants() {
            return anotherParticipants;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            TransferMetaHistoryDto that = (TransferMetaHistoryDto) o;
            return Objects.equals(transferRequestId, that.transferRequestId) &&
                    roleInTransfer == that.roleInTransfer &&
                    Objects.equals(anotherParticipants, that.anotherParticipants);
        }

        @Override
        public int hashCode() {
            return Objects.hash(transferRequestId, roleInTransfer, anotherParticipants);
        }

        @Override
        public String toString() {
            return ReflectionToStringBuilder.toString(this);
        }

        public static final class AnotherDto {
            private final Long serviceId;
            private final String folderId;
            private final String accountId;

            @JsonCreator
            public AnotherDto(
                    Long serviceId,
                    String folderId,
                    String accountId
            ) {
                this.serviceId = serviceId;
                this.folderId = folderId;
                this.accountId = accountId;
            }

            @Schema(description = "Another participant service ID.", required = true)
            public Long getServiceId() {
                return serviceId;
            }

            @Schema(description = "Another participant folder ID.", required = true)
            public String getFolderId() {
                return folderId;
            }

            @Schema(description = "Another participant account ID (not required).")
            public String getAccountId() {
                return accountId;
            }

            @Override
            public boolean equals(Object o) {
                if (this == o) {
                    return true;
                }
                if (o == null || getClass() != o.getClass()) {
                    return false;
                }
                AnotherDto that = (AnotherDto) o;
                return Objects.equals(serviceId, that.serviceId) &&
                        Objects.equals(folderId, that.folderId) &&
                        Objects.equals(accountId, that.accountId);
            }

            @Override
            public int hashCode() {
                return Objects.hash(serviceId, folderId, accountId);
            }

            @Override
            public String toString() {
                return ReflectionToStringBuilder.toString(this);
            }
        }
    }
}
