package ru.yandex.intranet.d.web.model.idm;

import java.util.Collections;
import java.util.Map;
import java.util.Objects;

import com.fasterxml.jackson.annotation.JsonAnyGetter;
import com.fasterxml.jackson.annotation.JsonIgnore;

import static ru.yandex.intranet.d.web.model.idm.IdmResponseCode.FAIL;
import static ru.yandex.intranet.d.web.model.idm.IdmResponseCode.OK;

public class IdmResult {
    private final int code;
    private final IdmMessageLevel state;
    private final String message;

    IdmResult(Builder builder) {
        this.code = builder.code.getCode();
        this.state = builder.state;
        this.message = builder.message;

        if (state != null || message != null) {
            Objects.requireNonNull(state, "State must be not null, when message is not null");
            Objects.requireNonNull(message, "Message must be not, when state is not null");
        }
    }

    public int getCode() {
        return code;
    }

    @JsonAnyGetter
    public Map<String, String> getMessage() {
        return state == null
                ? Collections.emptyMap()
                : Collections.singletonMap(state.getName(), message);
    }

    public static Builder builder() {
        return new Builder();
    }

    public static IdmResult newOkIdmResult() {
        return builder()
                .code(OK)
                .build();
    }

    public static IdmResult newWarningMessageOkIdmResult(String message) {
        return builder()
                .code(OK)
                .state(IdmMessageLevel.WARNING)
                .message(message)
                .build();
    }

    public static IdmResult newFatalMessageIdmResult(String message) {
        return newFailIdmResult(IdmMessageLevel.FATAL, message);
    }

    public static IdmResult newErrorMessageIdmResult(String message) {
        return newFailIdmResult(IdmMessageLevel.ERROR, message);
    }

    public static IdmResult newFailIdmResult(IdmMessageLevel state, String message) {
        return builder()
                .code(FAIL)
                .state(state)
                .message(message)
                .build();
    }

    @JsonIgnore
    public static IdmResult fromThrowable(Throwable t) {
        return newErrorMessageIdmResult(t.getMessage());
    }

    public static class Builder {
        private IdmResponseCode code;
        private IdmMessageLevel state;
        private String message;

        public Builder code(IdmResponseCode code) {
            this.code = code;
            return this;
        }

        public Builder message(String message) {
            this.message = message;
            return this;
        }

        public Builder state(IdmMessageLevel state) {
            this.state = state;
            return this;
        }

        public IdmResult build() {
            return new IdmResult(this);
        }
    }
}
