package ru.yandex.intranet.d.web.model.imports;

import java.util.List;
import java.util.Objects;
import java.util.Optional;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import io.swagger.v3.oas.annotations.media.Schema;

/**
 * Resource identity by {@link String} resource type key, set of {@link SegmentKey} and {@link AccountSpaceIdentityDto}.
 *
 * @author Ruslan Kadriev <aqru@yandex-team.ru>
 * @since 1-02-2021
 */
@Schema(description = "Resource identity by resource type key, set of segments and account space identity.")
@JsonIgnoreProperties(ignoreUnknown = true)
public final class ResourceIdentityDto {
    private final String resourceTypeKey;
    private final AccountSpaceIdentityDto accountSpaceIdentityDto;
    private final List<SegmentKey> segments;

    @JsonCreator
    public ResourceIdentityDto(String resourceTypeKey, AccountSpaceIdentityDto accountSpaceIdentityDto,
                               List<SegmentKey> segments) {
        this.resourceTypeKey = resourceTypeKey;
        this.accountSpaceIdentityDto = accountSpaceIdentityDto;
        this.segments = segments;
    }

    @Schema(description = "String resource type key.", required = true)
    public Optional<String> getResourceTypeKey() {
        return Optional.ofNullable(resourceTypeKey);
    }

    @Schema(description = "Account space identity.")
    public Optional<AccountSpaceIdentityDto> getAccountSpaceIdentityDto() {
        return Optional.ofNullable(accountSpaceIdentityDto);
    }

    @Schema(description = "Set of resource keys of segments with its segmentations.")
    public Optional<List<SegmentKey>> getSegments() {
        return Optional.ofNullable(segments);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        ResourceIdentityDto that = (ResourceIdentityDto) o;
        return Objects.equals(resourceTypeKey, that.resourceTypeKey) &&
                Objects.equals(accountSpaceIdentityDto, that.accountSpaceIdentityDto) &&
                Objects.equals(segments, that.segments);
    }

    @Override
    public int hashCode() {
        return Objects.hash(resourceTypeKey, accountSpaceIdentityDto, segments);
    }

    @Override
    public String toString() {
        return "ResourceIdentityDto{" +
                "resourceTypeKey='" + resourceTypeKey + '\'' +
                ", accountSpaceIdentityDto=" + accountSpaceIdentityDto +
                ", segments=" + segments +
                '}';
    }
}
