package ru.yandex.intranet.d.web.model.loans.front

import com.fasterxml.jackson.annotation.JsonCreator
import com.fasterxml.jackson.annotation.JsonFormat
import com.fasterxml.jackson.annotation.JsonIgnoreProperties
import io.swagger.v3.oas.annotations.media.Schema
import ru.yandex.intranet.d.kotlin.LoanId
import ru.yandex.intranet.d.kotlin.TransferRequestId
import ru.yandex.intranet.d.model.loans.LoanModel
import ru.yandex.intranet.d.model.loans.LoanStatus
import ru.yandex.intranet.d.model.loans.LoanType
import ru.yandex.intranet.d.web.model.loans.LoanActionSubjectDto
import ru.yandex.intranet.d.web.model.loans.LoanActionSubjectsDto
import ru.yandex.intranet.d.web.model.loans.LoanAmountsDto
import ru.yandex.intranet.d.web.model.loans.LoanDueDateDto
import ru.yandex.intranet.d.web.model.loans.LoanSubjectDto
import java.time.Instant

@JsonIgnoreProperties(ignoreUnknown = true)
data class FrontLoanDto @JsonCreator constructor(
    @get:Schema(description = "Loan ID", nullable = false)
    val id: LoanId,

    @get:Schema(description = "Loan status", nullable = false)
    val status: LoanStatus,

    @get:Schema(description = "Loan type", nullable = false)
    val type: LoanType,

    @get:Schema(description = "Loan creation time", nullable = false)
    @JsonFormat(shape = JsonFormat.Shape.STRING, pattern = "yyyy-MM-dd'T'HH:mm:ss.SSS'Z'", timezone = "UTC")
    val createdAt: Instant,

    @get:Schema(description = "Loan is due at, local date and timezone", nullable = false)
    val dueAt: LoanDueDateDto,

    @get:Schema(description = "Loan settlement time", nullable = true)
    @JsonFormat(shape = JsonFormat.Shape.STRING, pattern = "yyyy-MM-dd'T'HH:mm:ss.SSS'Z'", timezone = "UTC")
    val settledAt: Instant?,

    @get:Schema(description = "Loan update time", nullable = true)
    @JsonFormat(shape = JsonFormat.Shape.STRING, pattern = "yyyy-MM-dd'T'HH:mm:ss.SSS'Z'", timezone = "UTC")
    val updatedAt: Instant?,

    @get:Schema(description = "Loan version", nullable = false)
    val version: Long,

    @get:Schema(description = "Loan was requested by", nullable = false)
    val requestedBy: LoanActionSubjectDto,

    @get:Schema(description = "Loan was approved by", nullable = false)
    val requestApprovedBy: LoanActionSubjectsDto,

    @get:Schema(description = "Transfer request ID of borrow", nullable = false)
    val borrowTransferRequestId: TransferRequestId,

    @get:Schema(description = "Borrow source", nullable = false)
    val borrowedFrom: LoanSubjectDto,

    @get:Schema(description = "Borrow destination", nullable = false)
    val borrowedTo: LoanSubjectDto,

    @get:Schema(description = "Pay off source", nullable = false)
    val payOffFrom: LoanSubjectDto,

    @get:Schema(description = "Pay off destination", nullable = false)
    val payOffTo: LoanSubjectDto,

    @get:Schema(description = "Borrowed amount", nullable = false)
    val borrowedAmounts: LoanAmountsDto,

    @get:Schema(description = "Pay off amounts", nullable = false)
    val payOffAmounts: LoanAmountsDto,

    @get:Schema(description = "Amount left to pay off", nullable = false)
    val dueAmounts: LoanAmountsDto
) {
    constructor(model: LoanModel) : this(
        id = model.id,
        status = model.status,
        type = model.type,
        createdAt = model.createdAt,
        dueAt = LoanDueDateDto(model.dueAt),
        settledAt = model.settledAt,
        updatedAt = model.updatedAt,
        version = model.version,
        requestedBy = LoanActionSubjectDto(model.requestedBy),
        requestApprovedBy = LoanActionSubjectsDto(model.requestApprovedBy),
        borrowTransferRequestId = model.borrowTransferRequestId,
        borrowedFrom = LoanSubjectDto(model.borrowedFrom),
        borrowedTo = LoanSubjectDto(model.borrowedTo),
        payOffFrom = LoanSubjectDto(model.payOffFrom),
        payOffTo = LoanSubjectDto(model.payOffTo),
        borrowedAmounts = LoanAmountsDto(model.borrowedAmounts),
        payOffAmounts = LoanAmountsDto(model.payOffAmounts),
        dueAmounts = LoanAmountsDto(model.dueAmounts)
    )
}
