package ru.yandex.intranet.d.web.model.operations;

import java.time.Instant;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonFormat;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.google.common.base.Preconditions;
import io.swagger.v3.oas.annotations.media.Schema;

/**
 * Operation DTO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Schema(description = "Operation.")
@JsonIgnoreProperties(ignoreUnknown = true)
public class OperationDto {

    private final String id;
    private final String providerId;
    private final String accountsSpaceId;
    private final Instant createdAt;
    private final OperationStatusDto status;
    private final OperationErrorsDto failure;
    private final List<OperationRequestLogDto> requestLogs;

    @JsonCreator
    @SuppressWarnings("ParameterNumber")
    public OperationDto(
            String id,
            String providerId,
            String accountsSpaceId,
            @JsonFormat(shape = JsonFormat.Shape.STRING, pattern = "yyyy-MM-dd'T'HH:mm:ss.SSS'Z'", timezone = "UTC")
            Instant createdAt,
            OperationStatusDto status,
            OperationErrorsDto failure,
            List<OperationRequestLogDto> requestLogs) {
        this.id = id;
        this.providerId = providerId;
        this.accountsSpaceId = accountsSpaceId;
        this.createdAt = createdAt;
        this.status = status;
        this.failure = failure;
        this.requestLogs = requestLogs;
    }

    public static Builder builder() {
        return new Builder();
    }

    @Schema(description = "Operation id.", required = true)
    public String getId() {
        return id;
    }

    @Schema(description = "Provider id.", required = true)
    public String getProviderId() {
        return providerId;
    }

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    @Schema(description = "Accounts space id.")
    public Optional<String> getAccountsSpaceId() {
        return Optional.ofNullable(accountsSpaceId);
    }

    @Schema(description = "Operation created at.", required = true)
    @JsonFormat(shape = JsonFormat.Shape.STRING, pattern = "yyyy-MM-dd'T'HH:mm:ss.SSS'Z'", timezone = "UTC")
    public Instant getCreatedAt() {
        return createdAt;
    }

    @Schema(description = "Operation status.", required = true)
    public OperationStatusDto getStatus() {
        return status;
    }

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    @Schema(description = "Operation error messages.")
    public Optional<OperationErrorsDto> getFailure() {
        return Optional.ofNullable(failure);
    }

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    @Schema(description = "Request logs")
    public Optional<List<OperationRequestLogDto>> getRequestLogs() {
        return Optional.ofNullable(requestLogs);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        OperationDto that = (OperationDto) o;
        return Objects.equals(id, that.id) &&
                Objects.equals(providerId, that.providerId) &&
                Objects.equals(accountsSpaceId, that.accountsSpaceId) &&
                Objects.equals(createdAt, that.createdAt) &&
                status == that.status &&
                Objects.equals(failure, that.failure);
    }

    @Override
    public int hashCode() {
        return Objects.hash(id, providerId, accountsSpaceId, createdAt, status, failure);
    }

    @Override
    public String toString() {
        return "OperationDto{" +
                "id='" + id + '\'' +
                ", providerId='" + providerId + '\'' +
                ", accountsSpaceId='" + accountsSpaceId + '\'' +
                ", createdAt=" + createdAt +
                ", status=" + status +
                ", failure=" + failure +
                '}';
    }

    public static final class Builder {

        private String id;
        private String providerId;
        private String accountsSpaceId;
        private Instant createdAt;
        private OperationStatusDto status;
        private OperationErrorsDto failure;
        private List<OperationRequestLogDto> requestLogs;

        private Builder() {
        }

        public Builder id(String id) {
            this.id = id;
            return this;
        }

        public Builder providerId(String providerId) {
            this.providerId = providerId;
            return this;
        }

        public Builder accountsSpaceId(String accountsSpaceId) {
            this.accountsSpaceId = accountsSpaceId;
            return this;
        }

        public Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public Builder status(OperationStatusDto status) {
            this.status = status;
            return this;
        }

        public Builder failure(OperationErrorsDto failure) {
            this.failure = failure;
            return this;
        }

        public Builder requestLogs(List<OperationRequestLogDto> requestLogs) {
            this.requestLogs = requestLogs;
            return this;
        }

        public OperationDto build() {
            Preconditions.checkNotNull(id, "Id is required");
            Preconditions.checkNotNull(providerId, "Provider id is required");
            Preconditions.checkNotNull(createdAt, "Created at is required");
            Preconditions.checkNotNull(status, "Status is required");
            return new OperationDto(id, providerId, accountsSpaceId, createdAt, status, failure, requestLogs);
        }

    }

}
