package ru.yandex.intranet.d.web.model.operations;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import io.swagger.v3.oas.annotations.media.Schema;

/**
 * Operation errors DTO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Schema(description = "Operation errors info.")
@JsonIgnoreProperties(ignoreUnknown = true)
public class OperationErrorsDto {

    private final Set<String> errors;
    private final Map<String, Set<String>> fieldErrors;
    private final Map<String, Set<Object>> details;

    @JsonCreator
    public OperationErrorsDto(Set<String> errors,
                              Map<String, Set<String>> fieldErrors,
                              Map<String, Set<Object>> details) {
        this.errors = errors;
        this.fieldErrors = fieldErrors;
        this.details = details;
    }

    public static Builder builder() {
        return new Builder();
    }

    @Schema(description = "Errors list.")
    public Set<String> getErrors() {
        return errors;
    }

    @Schema(description = "Errors per field.")
    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    public Map<String, Set<String>> getFieldErrors() {
        return fieldErrors;
    }

    @Schema(description = "Error details.")
    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    public Map<String, Set<Object>> getDetails() {
        return details;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        OperationErrorsDto that = (OperationErrorsDto) o;
        return Objects.equals(errors, that.errors) &&
                Objects.equals(fieldErrors, that.fieldErrors) &&
                Objects.equals(details, that.details);
    }

    @Override
    public int hashCode() {
        return Objects.hash(errors, fieldErrors, details);
    }

    @Override
    public String toString() {
        return "OperationErrorsDto{" +
                "errors=" + errors +
                ", fieldErrors=" + fieldErrors +
                ", details=" + details +
                '}';
    }

    public static final class Builder {

        private final Set<String> errors = new HashSet<>();
        private final Map<String, Set<String>> fieldErrors = new HashMap<>();
        private final Map<String, Set<Object>> details = new HashMap<>();

        private Builder() {
        }

        public Builder addError(String error) {
            Objects.requireNonNull(error, "Error is required.");
            errors.add(error);
            return this;
        }

        public Builder addFieldError(String field, String error) {
            Objects.requireNonNull(field, "Field is required.");
            Objects.requireNonNull(error, "Error is required.");
            fieldErrors.computeIfAbsent(field, k -> new HashSet<>()).add(error);
            return this;
        }

        public Builder addDetail(String key, Object value) {
            Objects.requireNonNull(key, "Key is required.");
            Objects.requireNonNull(value, "Value is required.");
            details.computeIfAbsent(key, k -> new HashSet<>()).add(value);
            return this;
        }

        public OperationErrorsDto build() {
            return new OperationErrorsDto(errors, fieldErrors, details);
        }

    }

}
