package ru.yandex.intranet.d.web.model.operations;

import java.time.Instant;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonFormat;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import io.swagger.v3.oas.annotations.media.Schema;

/**
 * @author Mustakayev Marat <mmarat248@yandex-team.ru>
 */
public class OperationRequestLogDto {

    private final String requestId;
    private final Instant createdAt;
    private final JsonNode requestData;
    private final JsonNode responseData;
    private final Instant expireAt;

    @JsonCreator
    @SuppressWarnings("ParameterNumber")
    public OperationRequestLogDto(
            String requestId,
            @JsonFormat(shape = JsonFormat.Shape.STRING, pattern = "yyyy-MM-dd'T'HH:mm:ss.SSS'Z'", timezone = "UTC")
            Instant createdAt,
            JsonNode requestData,
            JsonNode responseData,
            Instant expireAt
    ) {
        this.requestId = requestId;
        this.createdAt = createdAt;
        this.requestData = requestData;
        this.responseData = responseData;
        this.expireAt = expireAt;
    }

    public static OperationRequestLogDto.Builder builder() {
        return new OperationRequestLogDto.Builder();
    }

    @Schema(description = "Request Id", required = true)
    public String getRequestId() {
        return requestId;
    }

    @Schema(description = "Operation created at.", required = true)
    @JsonFormat(shape = JsonFormat.Shape.STRING, pattern = "yyyy-MM-dd'T'HH:mm:ss.SSS'Z'", timezone = "UTC")
    public Instant getCreatedAt() {
        return createdAt;
    }

    @Schema(description = "Request data", required = true)
    public JsonNode getRequestData() {
        return requestData;
    }

    @Schema(description = "Response data", required = true)
    public JsonNode getResponseData() {
        return responseData;
    }

    @Schema(description = "Operation expire at.", required = true)
    @JsonFormat(shape = JsonFormat.Shape.STRING, pattern = "yyyy-MM-dd'T'HH:mm:ss.SSS'Z'", timezone = "UTC")
    public Instant getExpireAt() {
        return expireAt;
    }

    public static final class Builder {

        private String requestId;
        private Instant createdAt;
        private JsonNode requestData;
        private JsonNode responseData;
        private Instant expireAt;

        private  final ObjectMapper mapper = new ObjectMapper();

        private Builder() {
        }

        public OperationRequestLogDto.Builder requestId(String requestId) {
            this.requestId = requestId;
            return this;
        }

        public OperationRequestLogDto.Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public OperationRequestLogDto.Builder requestData(String requestData) {
            try {
                this.requestData = mapper.readTree(requestData);
            } catch (JsonProcessingException ignored) { }
            return this;
        }

        public OperationRequestLogDto.Builder responseData(String responseData) {
            try {
                this.responseData = mapper.readTree(responseData);
            } catch (JsonProcessingException ignored) { }
            return this;
        }

        public OperationRequestLogDto.Builder expireAt(Instant expireAt) {
            this.expireAt = expireAt;
            return this;
        }

        public OperationRequestLogDto build() {
            return new OperationRequestLogDto(requestId, createdAt, requestData, responseData, expireAt);
        }

    }

}
