package ru.yandex.intranet.d.web.model.providers;

import java.time.Duration;
import java.time.Instant;
import java.util.List;
import java.util.Objects;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonFormat;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import io.swagger.v3.oas.annotations.media.Schema;
import org.apache.commons.lang3.builder.ReflectionToStringBuilder;

import ru.yandex.intranet.d.model.sync.ProvidersSyncStatusModel;

/**
 * Provider sync status.
 *
 * @author Vladimir Zaytsev <vzay@yandex-team.ru>
 * @since 18-06-2021
 */
@Schema(description = "Provider sync status.")
@JsonIgnoreProperties(ignoreUnknown = true)
public final class ProviderSyncStatusDto {
    private final Instant lastSyncStart;
    private final Instant lastSyncFinish;
    private final SyncStatuses lastSyncStatus;
    private final Long accountsCount;
    private final Long quotasCount;
    private final Duration syncDuration;
    private final List<ProviderSyncErrorDto> providerSyncErrors;

    @SuppressWarnings("ParameterNumber")
    @JsonCreator
    public ProviderSyncStatusDto(
            Instant lastSyncStart,
            Instant lastSyncFinish,
            SyncStatuses lastSyncStatus,
            Long accountsCount,
            Long quotasCount,
            Duration syncDuration,
            List<ProviderSyncErrorDto> providerSyncErrors) {
        this.lastSyncStart = lastSyncStart;
        this.lastSyncFinish = lastSyncFinish;
        this.lastSyncStatus = lastSyncStatus;
        this.accountsCount = accountsCount;
        this.quotasCount = quotasCount;
        this.syncDuration = syncDuration;
        this.providerSyncErrors = providerSyncErrors;
    }

    @Schema(description = "Last sync start timestamp in UTC.", required = false)
    @JsonFormat(shape = JsonFormat.Shape.STRING, pattern = "yyyy-MM-dd'T'HH:mm:ss.SSS'Z'", timezone = "UTC")
    public Instant getLastSyncStart() {
        return lastSyncStart;
    }

    @Schema(description = "Last sync finish timestamp in UTC.", required = false)
    @JsonFormat(shape = JsonFormat.Shape.STRING, pattern = "yyyy-MM-dd'T'HH:mm:ss.SSS'Z'", timezone = "UTC")
    public Instant getLastSyncFinish() {
        return lastSyncFinish;
    }

    @Schema(description = "Last sync finish status.", required = false)
    public SyncStatuses getLastSyncStatus() {
        return lastSyncStatus;
    }

    @Schema(description = "Last sync accounts count.", required = false)
    public Long getAccountsCount() {
        return accountsCount;
    }

    @Schema(description = "Last sync quotas count.", required = false)
    public Long getQuotasCount() {
        return quotasCount;
    }

    @Schema(description = "Last sync duration.", required = false)
    public Duration getSyncDuration() {
        return syncDuration;
    }

    @Schema(description = "Sync errors, if any occurred.", required = false)
    public List<ProviderSyncErrorDto> getProviderSyncErrors() {
        return providerSyncErrors;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        ProviderSyncStatusDto that = (ProviderSyncStatusDto) o;
        return Objects.equals(lastSyncStart, that.lastSyncStart) &&
                Objects.equals(lastSyncFinish, that.lastSyncFinish) &&
                lastSyncStatus == that.lastSyncStatus &&
                Objects.equals(accountsCount, that.accountsCount) &&
                Objects.equals(quotasCount, that.quotasCount) &&
                Objects.equals(syncDuration, that.syncDuration) &&
                Objects.equals(providerSyncErrors, that.providerSyncErrors);
    }

    @Override
    public String toString() {
        return ReflectionToStringBuilder.toString(this);
    }

    @Override
    public int hashCode() {
        return Objects.hash(lastSyncStart, lastSyncFinish, lastSyncStatus, accountsCount, quotasCount, syncDuration,
                providerSyncErrors);
    }

    public enum SyncStatuses {
        RUNNING,
        DONE_OK,
        DONE_ERROR;

        public static SyncStatuses from(ProvidersSyncStatusModel.SyncStatuses status) {
            if (status == null) {
                return null;
            }
            switch (status) {
                case RUNNING:
                    return RUNNING;
                case DONE_OK:
                    return DONE_OK;
                case DONE_ERROR:
                    return DONE_ERROR;
                default:
                    throw new IllegalStateException("Unknown SyncStatuses value:" + status);
            }
        }
    }
}
