package ru.yandex.intranet.d.web.model.provisions;

import java.math.BigDecimal;
import java.util.Objects;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import io.swagger.v3.oas.annotations.media.Schema;

/**
 * Account provision DTO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Schema(description = "Account provision.")
@JsonIgnoreProperties(ignoreUnknown = true)
public class AccountProvisionDto {

    private final String folderId;
    private final String accountId;
    private final String providerId;
    private final String resourceId;
    private final BigDecimal provided;
    private final String providedUnitKey;
    private final BigDecimal allocated;
    private final String allocatedUnitKey;

    @JsonCreator
    @SuppressWarnings("ParameterNumber")
    public AccountProvisionDto(String folderId,
                               String accountId,
                               String providerId,
                               String resourceId,
                               BigDecimal provided,
                               String providedUnitKey,
                               BigDecimal allocated,
                               String allocatedUnitKey) {
        this.folderId = folderId;
        this.accountId = accountId;
        this.providerId = providerId;
        this.resourceId = resourceId;
        this.provided = provided;
        this.providedUnitKey = providedUnitKey;
        this.allocated = allocated;
        this.allocatedUnitKey = allocatedUnitKey;
    }

    public static Builder builder() {
        return new Builder();
    }

    @Schema(description = "Folder id.", required = true)
    public String getFolderId() {
        return folderId;
    }

    @Schema(description = "Account id.", required = true)
    public String getAccountId() {
        return accountId;
    }

    @Schema(description = "Provider id.", required = true)
    public String getProviderId() {
        return providerId;
    }

    @Schema(description = "Resource id.", required = true)
    public String getResourceId() {
        return resourceId;
    }

    @Schema(description = "Provided amount.", required = true)
    public BigDecimal getProvided() {
        return provided;
    }

    @Schema(description = "Provided amount unit key.", required = true)
    public String getProvidedUnitKey() {
        return providedUnitKey;
    }

    @Schema(description = "Allocated amount.", required = true)
    public BigDecimal getAllocated() {
        return allocated;
    }

    @Schema(description = "Allocated amount unit key.", required = true)
    public String getAllocatedUnitKey() {
        return allocatedUnitKey;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        AccountProvisionDto that = (AccountProvisionDto) o;
        return Objects.equals(folderId, that.folderId) &&
                Objects.equals(accountId, that.accountId) &&
                Objects.equals(providerId, that.providerId) &&
                Objects.equals(resourceId, that.resourceId) &&
                Objects.equals(provided, that.provided) &&
                Objects.equals(providedUnitKey, that.providedUnitKey) &&
                Objects.equals(allocated, that.allocated) &&
                Objects.equals(allocatedUnitKey, that.allocatedUnitKey);
    }

    @Override
    public int hashCode() {
        return Objects.hash(folderId, accountId, providerId, resourceId, provided, providedUnitKey,
                allocated, allocatedUnitKey);
    }

    @Override
    public String toString() {
        return "AccountProvisionDto{" +
                "folderId='" + folderId + '\'' +
                ", accountId='" + accountId + '\'' +
                ", providerId='" + providerId + '\'' +
                ", resourceId='" + resourceId + '\'' +
                ", provided=" + provided +
                ", providedUnitKey='" + providedUnitKey + '\'' +
                ", allocated=" + allocated +
                ", allocatedUnitKey='" + allocatedUnitKey + '\'' +
                '}';
    }

    public static final class Builder {

        private String folderId;
        private String accountId;
        private String providerId;
        private String resourceId;
        private BigDecimal provided;
        private String providedUnitKey;
        private BigDecimal allocated;
        private String allocatedUnitKey;

        private Builder() {
        }

        public Builder folderId(String folderId) {
            this.folderId = folderId;
            return this;
        }

        public Builder accountId(String accountId) {
            this.accountId = accountId;
            return this;
        }

        public Builder providerId(String providerId) {
            this.providerId = providerId;
            return this;
        }

        public Builder resourceId(String resourceId) {
            this.resourceId = resourceId;
            return this;
        }

        public Builder provided(BigDecimal provided) {
            this.provided = provided;
            return this;
        }

        public Builder providedUnitKey(String providedUnitKey) {
            this.providedUnitKey = providedUnitKey;
            return this;
        }

        public Builder allocated(BigDecimal allocated) {
            this.allocated = allocated;
            return this;
        }

        public Builder allocatedUnitKey(String allocatedUnitKey) {
            this.allocatedUnitKey = allocatedUnitKey;
            return this;
        }

        public AccountProvisionDto build() {
            Objects.requireNonNull(folderId, "Folder id is required");
            Objects.requireNonNull(accountId, "Account id is required");
            Objects.requireNonNull(providerId, "Provider id is required");
            Objects.requireNonNull(resourceId, "Resource id is required");
            Objects.requireNonNull(provided, "Provided is required");
            Objects.requireNonNull(providedUnitKey, "Provided unit key is required");
            Objects.requireNonNull(allocated, "Allocated is required");
            Objects.requireNonNull(allocatedUnitKey, "Allocated unit key is required");
            return new AccountProvisionDto(folderId, accountId, providerId, resourceId, provided, providedUnitKey,
                    allocated, allocatedUnitKey);
        }

    }

}
