package ru.yandex.intranet.d.web.model.provisions;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import io.swagger.v3.oas.annotations.media.Schema;

/**
 * Update provision operation DTO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Schema(description = "Update provision operation result.")
@JsonIgnoreProperties(ignoreUnknown = true)
public final class UpdateProvisionOperationDto {

    private final UpdateProvisionOperationStatusDto operationStatus;
    private final List<AccountProvisionDto> result;
    private final String operationId;

    @JsonCreator
    public UpdateProvisionOperationDto(UpdateProvisionOperationStatusDto operationStatus,
                                       List<AccountProvisionDto> result,
                                       String operationId) {
        this.operationStatus = operationStatus;
        this.result = result != null ? result : List.of();
        this.operationId = operationId;
    }

    public static Builder builder() {
        return new Builder();
    }

    @Schema(description = "Operation status.", required = true)
    public UpdateProvisionOperationStatusDto getOperationStatus() {
        return operationStatus;
    }

    @Schema(description = "Updated provisions.")
    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    public List<AccountProvisionDto> getResult() {
        return result;
    }

    @Schema(description = "Unique operation id.", required = true)
    public String getOperationId() {
        return operationId;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        UpdateProvisionOperationDto that = (UpdateProvisionOperationDto) o;
        return operationStatus == that.operationStatus &&
                Objects.equals(result, that.result) &&
                Objects.equals(operationId, that.operationId);
    }

    @Override
    public int hashCode() {
        return Objects.hash(operationStatus, result, operationId);
    }

    @Override
    public String toString() {
        return "UpdateProvisionOperationDto{" +
                "operationStatus=" + operationStatus +
                ", result=" + result +
                ", operationId='" + operationId + '\'' +
                '}';
    }

    public static final class Builder {

        private final List<AccountProvisionDto> result = new ArrayList<>();

        private UpdateProvisionOperationStatusDto operationStatus;
        private String operationId;

        private Builder() {
        }

        public Builder operationStatus(UpdateProvisionOperationStatusDto operationStatus) {
            this.operationStatus = operationStatus;
            return this;
        }

        public Builder operationId(String operationId) {
            this.operationId = operationId;
            return this;
        }

        public Builder addProvision(AccountProvisionDto provision) {
            this.result.add(provision);
            return this;
        }

        public Builder addProvisions(AccountProvisionDto... provisions) {
            this.result.addAll(Arrays.asList(provisions));
            return this;
        }

        public Builder addProvisions(Collection<? extends AccountProvisionDto> provisions) {
            this.result.addAll(provisions);
            return this;
        }

        public UpdateProvisionOperationDto build() {
            return new UpdateProvisionOperationDto(operationStatus, result, operationId);
        }

    }

}
