package ru.yandex.intranet.d.web.model.recipe;

import java.time.Instant;
import java.util.Objects;
import java.util.Optional;

import javax.annotation.Nullable;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonFormat;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import io.swagger.v3.oas.annotations.media.Schema;
import ru.yandex.intranet.d.web.model.accounts.AccountReserveTypeDto;

/**
 * Account.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Schema(description = "Account.")
@JsonIgnoreProperties(ignoreUnknown = true)
public final class AccountDto {

    private final String id;
    private final String providerId;
    private final String externalId;
    private final String externalKey;
    private final String folderId;
    private final String displayName;
    private final boolean deleted;
    private final Instant lastAccountUpdate;
    private final Long lastReceivedVersion;
    private final String latestSuccessfulAccountOperationId;
    private final String accountsSpacesId;
    private final boolean freeTier;
    @Nullable
    private final AccountReserveTypeDto reserveType;

    @JsonCreator
    @SuppressWarnings("ParameterNumber")
    public AccountDto(String id,
                      String providerId,
                      String externalId,
                      String externalKey,
                      String folderId,
                      String displayName,
                      boolean deleted,
                      @JsonFormat(shape = JsonFormat.Shape.STRING,
                              pattern = "yyyy-MM-dd'T'HH:mm:ss.SSS'Z'", timezone = "UTC")
                      Instant lastAccountUpdate,
                      Long lastReceivedVersion,
                      String latestSuccessfulAccountOperationId,
                      String accountsSpacesId,
                      boolean freeTier,
                      @Nullable AccountReserveTypeDto reserveType) {
        this.id = id;
        this.providerId = providerId;
        this.externalId = externalId;
        this.externalKey = externalKey;
        this.folderId = folderId;
        this.displayName = displayName;
        this.deleted = deleted;
        this.lastAccountUpdate = lastAccountUpdate;
        this.lastReceivedVersion = lastReceivedVersion;
        this.latestSuccessfulAccountOperationId = latestSuccessfulAccountOperationId;
        this.accountsSpacesId = accountsSpacesId;
        this.freeTier = freeTier;
        this.reserveType = reserveType;
    }

    @Schema(description = "Id.", required = true)
    public String getId() {
        return id;
    }

    @Schema(description = "Provider id.", required = true)
    public String getProviderId() {
        return providerId;
    }

    @Schema(description = "External id.", required = true)
    public String getExternalId() {
        return externalId;
    }

    @Schema(description = "External key.")
    public Optional<String> getExternalKey() {
        return Optional.ofNullable(externalKey);
    }

    @Schema(description = "Folder id.", required = true)
    public String getFolderId() {
        return folderId;
    }

    @Schema(description = "Display name.")
    public Optional<String> getDisplayName() {
        return Optional.ofNullable(displayName);
    }

    @Schema(description = "'Is deleted' flag.", required = true)
    public boolean isDeleted() {
        return deleted;
    }

    @Schema(description = "Last account update.")
    @JsonFormat(shape = JsonFormat.Shape.STRING, pattern = "yyyy-MM-dd'T'HH:mm:ss.SSS'Z'", timezone = "UTC")
    public Optional<Instant> getLastAccountUpdate() {
        return Optional.ofNullable(lastAccountUpdate);
    }

    @Schema(description = "Last received version.")
    public Optional<Long> getLastReceivedVersion() {
        return Optional.ofNullable(lastReceivedVersion);
    }

    @Schema(description = "Latest successful account operation id.")
    public Optional<String> getLatestSuccessfulAccountOperationId() {
        return Optional.ofNullable(latestSuccessfulAccountOperationId);
    }

    @Schema(description = "Accounts space id.")
    public Optional<String> getAccountsSpacesId() {
        return Optional.ofNullable(accountsSpacesId);
    }

    @Schema(description = "'Is free tier' flag.", required = true)
    public boolean isFreeTier() {
        return freeTier;
    }

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    @Schema(description = "Reserve type.")
    public Optional<AccountReserveTypeDto> getReserveType() {
        return Optional.ofNullable(reserveType);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        AccountDto that = (AccountDto) o;
        return deleted == that.deleted &&
                freeTier == that.freeTier &&
                Objects.equals(id, that.id) &&
                Objects.equals(providerId, that.providerId) &&
                Objects.equals(externalId, that.externalId) &&
                Objects.equals(externalKey, that.externalKey) &&
                Objects.equals(folderId, that.folderId) &&
                Objects.equals(displayName, that.displayName) &&
                Objects.equals(lastAccountUpdate, that.lastAccountUpdate) &&
                Objects.equals(lastReceivedVersion, that.lastReceivedVersion) &&
                Objects.equals(latestSuccessfulAccountOperationId, that.latestSuccessfulAccountOperationId) &&
                Objects.equals(accountsSpacesId, that.accountsSpacesId) &&
                Objects.equals(reserveType, that.reserveType);
    }

    @Override
    public int hashCode() {
        return Objects.hash(id, providerId, externalId, externalKey, folderId, displayName, deleted,
                lastAccountUpdate, lastReceivedVersion, latestSuccessfulAccountOperationId, accountsSpacesId,
                freeTier, reserveType);
    }

}
