package ru.yandex.intranet.d.web.model.recipe;

import java.time.Instant;
import java.util.Objects;
import java.util.Optional;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonFormat;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import io.swagger.v3.oas.annotations.media.Schema;

/**
 * Account quota.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Schema(description = "Account quota.")
@JsonIgnoreProperties(ignoreUnknown = true)
public final class AccountQuotaDto {

    private final String accountId;
    private final String resourceId;
    private final long providedQuota;
    private final long allocatedQuota;
    private final long frozenProvidedQuota;
    private final String folderId;
    private final String providerId;
    private final Instant lastProvisionUpdate;
    private final Long lastReceivedProvisionVersion;
    private final String latestSuccessfulProvisionOperationId;
    private final String providedQuotaUnitKey;
    private final String allocatedQuotaUnitKey;
    private final String frozenProvidedQuotaUnitKey;
    private final String externalAccountId;
    private final String accountsSpaceId;

    @JsonCreator
    @SuppressWarnings("ParameterNumber")
    public AccountQuotaDto(String accountId,
                           String resourceId,
                           long providedQuota,
                           long allocatedQuota,
                           long frozenProvidedQuota,
                           String folderId,
                           String providerId,
                           @JsonFormat(shape = JsonFormat.Shape.STRING,
                                   pattern = "yyyy-MM-dd'T'HH:mm:ss.SSS'Z'", timezone = "UTC")
                                       Instant lastProvisionUpdate,
                           Long lastReceivedProvisionVersion,
                           String latestSuccessfulProvisionOperationId,
                           String providedQuotaUnitKey,
                           String allocatedQuotaUnitKey,
                           String frozenProvidedQuotaUnitKey, String externalAccountId,
                           String accountsSpaceId) {
        this.accountId = accountId;
        this.resourceId = resourceId;
        this.providedQuota = providedQuota;
        this.allocatedQuota = allocatedQuota;
        this.frozenProvidedQuota = frozenProvidedQuota;
        this.folderId = folderId;
        this.providerId = providerId;
        this.lastProvisionUpdate = lastProvisionUpdate;
        this.lastReceivedProvisionVersion = lastReceivedProvisionVersion;
        this.latestSuccessfulProvisionOperationId = latestSuccessfulProvisionOperationId;
        this.providedQuotaUnitKey = providedQuotaUnitKey;
        this.allocatedQuotaUnitKey = allocatedQuotaUnitKey;
        this.frozenProvidedQuotaUnitKey = frozenProvidedQuotaUnitKey;
        this.externalAccountId = externalAccountId;
        this.accountsSpaceId = accountsSpaceId;
    }

    @Schema(description = "Account id.", required = true)
    public String getAccountId() {
        return accountId;
    }

    @Schema(description = "Resource id.", required = true)
    public String getResourceId() {
        return resourceId;
    }

    @Schema(description = "Provided quota amount.", required = true)
    public long getProvidedQuota() {
        return providedQuota;
    }

    @Schema(description = "Allocated quota amount.", required = true)
    public long getAllocatedQuota() {
        return allocatedQuota;
    }

    @Schema(description = "Frozen provided quota amount.", required = true)
    public long getFrozenProvidedQuota() {
        return frozenProvidedQuota;
    }

    @Schema(description = "Folder id.", required = true)
    public String getFolderId() {
        return folderId;
    }

    @Schema(description = "Provider id.", required = true)
    public String getProviderId() {
        return providerId;
    }

    @Schema(description = "Last provision update timestamp.")
    @JsonFormat(shape = JsonFormat.Shape.STRING, pattern = "yyyy-MM-dd'T'HH:mm:ss.SSS'Z'", timezone = "UTC")
    public Optional<Instant> getLastProvisionUpdate() {
        return Optional.ofNullable(lastProvisionUpdate);
    }

    @Schema(description = "Last received provision version.")
    public Optional<Long> getLastReceivedProvisionVersion() {
        return Optional.ofNullable(lastReceivedProvisionVersion);
    }

    @Schema(description = "Latest successful provision operation id.")
    public Optional<String> getLatestSuccessfulProvisionOperationId() {
        return Optional.ofNullable(latestSuccessfulProvisionOperationId);
    }

    @Schema(description = "Provided quota unit key.", required = true)
    public String getProvidedQuotaUnitKey() {
        return providedQuotaUnitKey;
    }

    @Schema(description = "Allocated quota unit key.", required = true)
    public String getAllocatedQuotaUnitKey() {
        return allocatedQuotaUnitKey;
    }

    @Schema(description = "Frozen provided quota unit key", required = true)
    public String getFrozenProvidedQuotaUnitKey() {
        return frozenProvidedQuotaUnitKey;
    }

    @Schema(description = "External account id.", required = true)
    public String getExternalAccountId() {
        return externalAccountId;
    }

    @Schema(description = "Accounts space id.")
    public Optional<String> getAccountsSpaceId() {
        return Optional.ofNullable(accountsSpaceId);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        AccountQuotaDto that = (AccountQuotaDto) o;
        return providedQuota == that.providedQuota &&
                allocatedQuota == that.allocatedQuota &&
                frozenProvidedQuota == that.frozenProvidedQuota &&
                Objects.equals(accountId, that.accountId) &&
                Objects.equals(resourceId, that.resourceId) &&
                Objects.equals(folderId, that.folderId) &&
                Objects.equals(providerId, that.providerId) &&
                Objects.equals(lastProvisionUpdate, that.lastProvisionUpdate) &&
                Objects.equals(lastReceivedProvisionVersion, that.lastReceivedProvisionVersion) &&
                Objects.equals(latestSuccessfulProvisionOperationId, that.latestSuccessfulProvisionOperationId) &&
                Objects.equals(providedQuotaUnitKey, that.providedQuotaUnitKey) &&
                Objects.equals(allocatedQuotaUnitKey, that.allocatedQuotaUnitKey) &&
                Objects.equals(frozenProvidedQuotaUnitKey, that.frozenProvidedQuotaUnitKey) &&
                Objects.equals(externalAccountId, that.externalAccountId) &&
                Objects.equals(accountsSpaceId, that.accountsSpaceId);
    }

    @Override
    public int hashCode() {
        return Objects.hash(accountId, resourceId, providedQuota, allocatedQuota, folderId, providerId,
                lastProvisionUpdate, lastReceivedProvisionVersion, latestSuccessfulProvisionOperationId,
                providedQuotaUnitKey, allocatedQuotaUnitKey, externalAccountId, accountsSpaceId, frozenProvidedQuota,
                frozenProvidedQuotaUnitKey);
    }

    @Override
    public String toString() {
        return "AccountQuotaDto{" +
                "accountId='" + accountId + '\'' +
                ", resourceId='" + resourceId + '\'' +
                ", providedQuota=" + providedQuota +
                ", allocatedQuota=" + allocatedQuota +
                ", frozenProvidedQuota=" + frozenProvidedQuota +
                ", folderId='" + folderId + '\'' +
                ", providerId='" + providerId + '\'' +
                ", lastProvisionUpdate=" + lastProvisionUpdate +
                ", lastReceivedProvisionVersion=" + lastReceivedProvisionVersion +
                ", latestSuccessfulProvisionOperationId='" + latestSuccessfulProvisionOperationId + '\'' +
                ", providedQuotaUnitKey='" + providedQuotaUnitKey + '\'' +
                ", allocatedQuotaUnitKey='" + allocatedQuotaUnitKey + '\'' +
                ", frozenProvidedQuotaUnitKey='" + frozenProvidedQuotaUnitKey + '\'' +
                ", externalAccountId='" + externalAccountId + '\'' +
                ", accountsSpaceId='" + accountsSpaceId + '\'' +
                '}';
    }

}
