package ru.yandex.intranet.d.web.model.resources;

import java.util.Objects;
import java.util.Optional;
import java.util.Set;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import io.swagger.v3.oas.annotations.media.Schema;

/**
 * Resource DTO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Schema(description = "Resource.")
@JsonIgnoreProperties(ignoreUnknown = true)
public final class ResourceDto {

    private final String id;
    private final String key;
    private final String providerId;
    private final String unitsEnsembleId;
    private final long version;
    private final String name;
    private final String description;
    private final boolean readOnly;
    private final boolean orderable;
    private final boolean managed;
    private final boolean virtual;
    private final Set<String> allowedUnitKeys;
    private final String defaultUnitKey;
    private final String providerApiUnitKey;
    private final InnerResourceTypeSegmentsDto segmentations;

    @JsonCreator
    @SuppressWarnings("ParameterNumber")
    public ResourceDto(String id,
                       String key,
                       String providerId,
                       String unitsEnsembleId,
                       long version,
                       String name,
                       String description,
                       boolean readOnly,
                       boolean orderable,
                       boolean managed,
                       boolean virtual,
                       Set<String> allowedUnitKeys,
                       String defaultUnitKey,
                       String providerApiUnitKey,
                       InnerResourceTypeSegmentsDto segmentations) {
        this.id = id;
        this.key = key;
        this.providerId = providerId;
        this.unitsEnsembleId = unitsEnsembleId;
        this.version = version;
        this.name = name;
        this.description = description;
        this.readOnly = readOnly;
        this.orderable = orderable;
        this.managed = managed;
        this.virtual = virtual;
        this.allowedUnitKeys = allowedUnitKeys;
        this.defaultUnitKey = defaultUnitKey;
        this.providerApiUnitKey = providerApiUnitKey;
        this.segmentations = segmentations;
    }

    @Schema(description = "Unique id.", required = true)
    public String getId() {
        return id;
    }

    @Schema(description = "Key, unique per provider.", required = true)
    public String getKey() {
        return key;
    }

    @Schema(description = "Provider id.", required = true)
    public String getProviderId() {
        return providerId;
    }

    @Schema(description = "Units ensemble id.", required = true)
    public String getUnitsEnsembleId() {
        return unitsEnsembleId;
    }

    @Schema(description = "Version.", required = true)
    public long getVersion() {
        return version;
    }

    @Schema(description = "Name.", required = true)
    public String getName() {
        return name;
    }

    @Schema(description = "Description.", required = true)
    public String getDescription() {
        return description;
    }

    @Schema(description = "'Read only' flag.", required = true)
    public boolean isReadOnly() {
        return readOnly;
    }

    @Schema(description = "'Resource can be ordered' flag.", required = true)
    public boolean isOrderable() {
        return orderable;
    }

    @Schema(description = "'Managed' flag.", required = true)
    public boolean isManaged() {
        return managed;
    }

    @Schema(description = "Allowed unit keys.", required = true)
    public Set<String> getAllowedUnitKeys() {
        return allowedUnitKeys;
    }

    @Schema(description = "Default unit key.", required = true)
    public String getDefaultUnitKey() {
        return defaultUnitKey;
    }

    @Schema(description = "Unit key, used to interact with provider.")
    public String getProviderApiUnitKey() {
        return providerApiUnitKey;
    }

    @Schema(description = "'Virtual' flag.", required = true)
    public boolean isVirtual() {
        return virtual;
    }

    @Schema(description = "Resource type and segments.")
    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    public Optional<InnerResourceTypeSegmentsDto> getSegmentations() {
        return Optional.ofNullable(segmentations);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        ResourceDto that = (ResourceDto) o;
        return version == that.version &&
                readOnly == that.readOnly &&
                orderable == that.orderable &&
                managed == that.managed &&
                virtual == that.virtual &&
                Objects.equals(id, that.id) &&
                Objects.equals(key, that.key) &&
                Objects.equals(providerId, that.providerId) &&
                Objects.equals(unitsEnsembleId, that.unitsEnsembleId) &&
                Objects.equals(name, that.name) &&
                Objects.equals(description, that.description) &&
                Objects.equals(allowedUnitKeys, that.allowedUnitKeys) &&
                Objects.equals(defaultUnitKey, that.defaultUnitKey) &&
                Objects.equals(segmentations, that.segmentations) &&
                Objects.equals(providerApiUnitKey, that.providerApiUnitKey);
    }

    @Override
    public int hashCode() {
        return Objects.hash(id, key, providerId, unitsEnsembleId, version, name, description, readOnly, virtual,
                orderable, managed, allowedUnitKeys, defaultUnitKey, segmentations, providerApiUnitKey);
    }

    @Override
    public String toString() {
        return "ResourceDto{" +
                "id='" + id + '\'' +
                ", key='" + key + '\'' +
                ", providerId='" + providerId + '\'' +
                ", unitsEnsembleId='" + unitsEnsembleId + '\'' +
                ", version=" + version +
                ", name='" + name + '\'' +
                ", description='" + description + '\'' +
                ", readOnly=" + readOnly +
                ", orderable=" + orderable +
                ", managed=" + managed +
                ", virtual=" + virtual +
                ", allowedUnitKeys=" + allowedUnitKeys +
                ", defaultUnitKey='" + defaultUnitKey + '\'' +
                ", providerApiUnitKey='" + providerApiUnitKey + '\'' +
                ", segmentations=" + segmentations +
                '}';
    }
}
