package ru.yandex.intranet.d.web.model.resources;

import java.util.Locale;
import java.util.Map;
import java.util.Objects;

import javax.annotation.Nullable;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import io.swagger.v3.oas.annotations.media.Schema;

import ru.yandex.intranet.d.i18n.Locales;
import ru.yandex.intranet.d.model.resources.ResourceSegmentSettingsModel;
import ru.yandex.intranet.d.model.resources.segmentations.ResourceSegmentationModel;
import ru.yandex.intranet.d.model.resources.segments.ResourceSegmentModel;

/**
 * Resource segmentation and segment DTO for front.
 *
 * @author Vladimir Zaytsev <vzay@yandex-team.ru>
 * @since 21-05-2021
 */
@Schema(description = "Resource segmentation and segment.")
@JsonIgnoreProperties(ignoreUnknown = true)
public final class ResourceSegmentationSegmentDto {
    private final String segmentationId;
    private final String segmentationName;
    private final Integer groupingOrder;
    private final String segmentId;
    private final String segmentName;
    private final Boolean uncommonSegment;

    @JsonCreator
    public ResourceSegmentationSegmentDto(
            String segmentationId,
            String segmentationName,
            Integer groupingOrder,
            String segmentId,
            String segmentName,
            Boolean uncommonSegment
    ) {
        this.segmentationId = segmentationId;
        this.segmentationName = segmentationName;
        this.groupingOrder = groupingOrder;
        this.segmentId = segmentId;
        this.segmentName = segmentName;
        this.uncommonSegment = uncommonSegment;
    }

    public ResourceSegmentationSegmentDto(
            ResourceSegmentSettingsModel model,
            @Nullable Map<String, ResourceSegmentationModel> segmentationsById,
            @Nullable Map<String, ResourceSegmentModel> segmentsById,
            @Nullable Locale locale
    ) {
        this.segmentationId = model.getSegmentationId();
        this.segmentId = model.getSegmentId();
        if (segmentationsById != null &&
                locale != null) {
            ResourceSegmentationModel segmentation = segmentationsById.get(model.getSegmentationId());
            this.segmentationName = Locales.select(segmentation.getNameEn(), segmentation.getNameRu(), locale);
            this.groupingOrder = segmentation.getGroupingOrder();
        } else {
            this.segmentationName = null;
            this.groupingOrder = null;
        }
        if (segmentsById != null &&
                locale != null) {
            ResourceSegmentModel segment = segmentsById.get(model.getSegmentId());
            this.segmentName = Locales.select(segment.getNameEn(), segment.getNameRu(), locale);
            this.uncommonSegment = segment.getUncommon().orElse(false);
        } else {
            this.segmentName = null;
            this.uncommonSegment = null;
        }
    }

    @Schema(description = "Segmentation id.", required = true)
    public String getSegmentationId() {
        return segmentationId;
    }

    @Schema(description = "Segmentation name.", required = false)
    public String getSegmentationName() {
        return segmentationName;
    }

    @Schema(description = "Segmentation grouping (and sorting) order.", required = false)
    public Integer getGroupingOrder() {
        return groupingOrder;
    }

    @Schema(description = "Segment id.", required = true)
    public String getSegmentId() {
        return segmentId;
    }

    @Schema(description = "Segment name.", required = false)
    public String getSegmentName() {
        return segmentName;
    }

    @Schema(description = "'Uncommon segment' flag")
    public Boolean getUncommonSegment() {
        return uncommonSegment;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        ResourceSegmentationSegmentDto that = (ResourceSegmentationSegmentDto) o;
        return Objects.equals(segmentationId, that.segmentationId) &&
                Objects.equals(segmentationName, that.segmentationName) &&
                Objects.equals(groupingOrder, that.groupingOrder) &&
                Objects.equals(segmentId, that.segmentId) &&
                Objects.equals(segmentName, that.segmentName) &&
                Objects.equals(uncommonSegment, that.uncommonSegment);
    }

    @Override
    public int hashCode() {
        return Objects.hash(segmentationId, segmentationName, groupingOrder, segmentId, segmentName, uncommonSegment);
    }

    @Override
    public String toString() {
        return "ResourceSegmentationSegmentDto{" +
                "segmentationId='" + segmentationId + '\'' +
                ", segmentationName='" + segmentationName + '\'' +
                ", groupingOrder=" + groupingOrder +
                ", segmentId='" + segmentId + '\'' +
                ", segmentName='" + segmentName + '\'' +
                ", uncommonSegment=" + uncommonSegment +
                '}';
    }
}
