package ru.yandex.intranet.d.web.model.resources;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

import javax.annotation.Nullable;

import com.fasterxml.jackson.annotation.JsonInclude;
import io.swagger.v3.oas.annotations.media.Schema;
import org.apache.commons.lang3.builder.ReflectionToStringBuilder;

import ru.yandex.intranet.d.services.resources.SelectionTreeResourceDto;
import ru.yandex.intranet.d.web.model.resources.directory.segmentations.SegmentationUISettingsDto;
import ru.yandex.intranet.d.web.model.units.front.FrontUnitsEnsembleDto;

/**
 * @author Mustakayev Marat <mmarat248@yandex-team.ru>
 */
@Schema(description = "Resource tree node.")
public class SelectionResourceTreeNodeDto {
    private final Map<String, SelectionResourceTreeNodeDto> childrenBySegmentId;
    private final String segmentationId;
    private final String segmentationKey;
    private final Set<String> accountsSpacesIds;
    private final List<SelectionTreeResourceDto> resources;
    private final @Nullable SegmentationUISettingsDto segmentationUISettings;
    private final String currentNodeSegmentId;
    private final String currentNodeSegmentKey;
    private final @Nullable Boolean currentNodeSegmentUncommon;
    private final @Nullable Map<String, FrontUnitsEnsembleDto> unitsEnsemblesById;

    private static final SelectionResourceTreeNodeDto EMPTY_NODE =
            new SelectionResourceTreeNodeDto(null, null, null, null, null, null, null, null, null, null);

    @SuppressWarnings("ParameterNumber")
    private SelectionResourceTreeNodeDto(Map<String, SelectionResourceTreeNodeDto> childrenBySegmentId,
                                         String segmentationId,
                                         String segmentationKey,
                                         Set<String> accountsSpacesIds,
                                         List<SelectionTreeResourceDto> resources,
                                         @Nullable SegmentationUISettingsDto segmentationUISettings,
                                         String currentNodeSegmentId,
                                         String currentNodeSegmentKey,
                                         @Nullable Boolean currentNodeSegmentUncommon,
                                         @Nullable Map<String, FrontUnitsEnsembleDto> unitsEnsemblesById
    ) {
        this.childrenBySegmentId = childrenBySegmentId;
        this.segmentationId = segmentationId;
        this.segmentationKey = segmentationKey;
        this.accountsSpacesIds = accountsSpacesIds;
        this.resources = resources;
        this.segmentationUISettings = segmentationUISettings;
        this.currentNodeSegmentId = currentNodeSegmentId;
        this.currentNodeSegmentKey = currentNodeSegmentKey;
        this.currentNodeSegmentUncommon = currentNodeSegmentUncommon;
        this.unitsEnsemblesById = unitsEnsemblesById;
    }

    @Schema(description = "Children of internal node (map by segments names). For terminal node has null value.")
    public Map<String, SelectionResourceTreeNodeDto> getChildrenBySegmentId() {
        return childrenBySegmentId;
    }

    @Schema(description = "Current node segmentation id. For terminal node has null value")
    public String getSegmentationId() {
        return segmentationId;
    }

    @Schema(description = "Current node segmentation key. For terminal node has null value")
    public String getSegmentationKey() {
        return segmentationKey;
    }

    @Schema(description = "Accounts spaces ids for terminal node. Null for internal node. " +
            "Empty if all resources in node have null accountsSpaceId value.")
    public Set<String> getAccountsSpacesIds() {
        return accountsSpacesIds;
    }

    @Schema(description = "Resources list for terminal node. For internal node has null value.")
    public List<SelectionTreeResourceDto> getResources() {
        return resources;
    }

    @Schema(description = "UI settings for current node segmentation. For terminal node has null value")
    public Optional<SegmentationUISettingsDto> getSegmentationUISettings() {
        return Optional.ofNullable(segmentationUISettings);
    }

    @Schema(description = "Current node segment id. Null for root node.")
    public String getCurrentNodeSegmentId() {
        return currentNodeSegmentId;
    }

    @Schema(description = "Current node segment key. Null for root node.")
    public String getCurrentNodeSegmentKey() {
        return currentNodeSegmentKey;
    }

    @Nullable
    @Schema(description = "Current node segment 'uncommon' flag. Null for root node.")
    public Boolean getCurrentNodeSegmentUncommon() {
        return currentNodeSegmentUncommon;
    }

    @Nullable
    @JsonInclude(JsonInclude.Include.NON_NULL)
    @Schema(description = "Units ensembles for all resources in the tree. This is present only in the root node.")
    public Map<String, FrontUnitsEnsembleDto> getUnitsEnsemblesById() {
        return unitsEnsemblesById;
    }

    public static SelectionResourceTreeNodeDto empty() {
        return EMPTY_NODE;
    }

    public static Builder builder() {
        return new Builder();
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        SelectionResourceTreeNodeDto that = (SelectionResourceTreeNodeDto) o;
        return Objects.equals(childrenBySegmentId, that.childrenBySegmentId) &&
                Objects.equals(segmentationId, that.segmentationId) &&
                Objects.equals(segmentationKey, that.segmentationKey) &&
                Objects.equals(accountsSpacesIds, that.accountsSpacesIds) &&
                Objects.equals(resources, that.resources) &&
                Objects.equals(segmentationUISettings, that.segmentationUISettings) &&
                Objects.equals(currentNodeSegmentId, that.currentNodeSegmentId) &&
                Objects.equals(currentNodeSegmentKey, that.currentNodeSegmentKey) &&
                Objects.equals(currentNodeSegmentUncommon, that.currentNodeSegmentUncommon) &&
                Objects.equals(unitsEnsemblesById, that.unitsEnsemblesById);
    }

    @Override
    public int hashCode() {
        return Objects.hash(childrenBySegmentId, segmentationId, segmentationKey, accountsSpacesIds, resources,
                segmentationUISettings, currentNodeSegmentId, currentNodeSegmentKey, currentNodeSegmentUncommon,
                unitsEnsemblesById);
    }

    @Override
    public String toString() {
        return ReflectionToStringBuilder.toString(this);
    }

    public static class Builder {
        private Map<String, SelectionResourceTreeNodeDto> childrenBySegmentId;
        private String segmentationId;
        private String segmentationKey;
        private Set<String> accountsSpacesIds;
        private List<SelectionTreeResourceDto> resources;
        private SegmentationUISettingsDto segmentationUISettings;
        private String currentNodeSegmentId;
        private String currentNodeSegmentKey;
        private Boolean currentNodeSegmentUncommon;
        private Map<String, FrontUnitsEnsembleDto> unitsEnsemblesById;

        public Builder childrenBySegmentId(Map<String, SelectionResourceTreeNodeDto> childrenBySegmentId) {
            this.childrenBySegmentId = childrenBySegmentId;
            return this;
        }

        public Builder segmentationId(String segmentationId) {
            this.segmentationId = segmentationId;
            return this;
        }

        public Builder segmentationKey(String segmentationKey) {
            this.segmentationKey = segmentationKey;
            return this;
        }

        public Builder accountsSpacesIds(Set<String> accountsSpacesIds) {
            this.accountsSpacesIds = accountsSpacesIds;
            return this;
        }

        public Builder resources(List<SelectionTreeResourceDto> resources) {
            this.resources = resources;
            return this;
        }

        public Builder segmentationUISettings(SegmentationUISettingsDto segmentationUISettings) {
            this.segmentationUISettings = segmentationUISettings;
            return this;
        }

        public Builder currentNodeSegmentId(String currentNodeSegmentId) {
            this.currentNodeSegmentId = currentNodeSegmentId;
            return this;
        }

        public Builder currentNodeSegmentKey(String currentNodeSegmentKey) {
            this.currentNodeSegmentKey = currentNodeSegmentKey;
            return this;
        }

        public Builder currentNodeSegmentUncommon(Boolean currentNodeSegmentUncommon) {
            this.currentNodeSegmentUncommon = currentNodeSegmentUncommon;
            return this;
        }

        public Builder unitsEnsemblesById(Map<String, FrontUnitsEnsembleDto> unitsEnsemblesById) {
            this.unitsEnsemblesById = unitsEnsemblesById;
            return this;
        }

        public SelectionResourceTreeNodeDto build() {
            return new SelectionResourceTreeNodeDto(childrenBySegmentId, segmentationId, segmentationKey,
                    accountsSpacesIds, resources, segmentationUISettings, currentNodeSegmentId, currentNodeSegmentKey,
                    currentNodeSegmentUncommon, unitsEnsemblesById);
        }
    }
}
