package ru.yandex.intranet.d.web.model.resources.directory.types;

import java.util.Objects;
import java.util.Optional;

import javax.annotation.Nullable;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import io.swagger.v3.oas.annotations.media.Schema;

import ru.yandex.intranet.d.util.AggregationSettingsHelper;
import ru.yandex.intranet.d.web.model.providers.AggregationSettingsInputDto;

/**
 * Create resource type DTO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Schema(description = "Resource type.")
@JsonIgnoreProperties(ignoreUnknown = true)
public final class ResourceTypeCreateDto {

    private final String key;
    private final String nameEn;
    private final String nameRu;
    private final String descriptionEn;
    private final String descriptionRu;
    private final String unitsEnsembleId;
    private final @Nullable Long sortingOrder;
    private final @Nullable AggregationSettingsInputDto aggregationSettings;

    @JsonCreator
    @SuppressWarnings("ParameterNumber")
    public ResourceTypeCreateDto(@JsonProperty("key") String key,
                                 @JsonProperty("nameEn") String nameEn,
                                 @JsonProperty("nameRu") String nameRu,
                                 @JsonProperty("descriptionEn") String descriptionEn,
                                 @JsonProperty("descriptionRu") String descriptionRu,
                                 String unitsEnsembleId,
                                 @JsonProperty("sortingOrder") @Nullable Long sortingOrder,
                                 @JsonProperty("aggregationSettings") @Nullable
                                 AggregationSettingsInputDto aggregationSettings) {
        this.key = key;
        this.nameEn = nameEn;
        this.nameRu = nameRu;
        this.descriptionEn = descriptionEn;
        this.descriptionRu = descriptionRu;
        this.unitsEnsembleId = unitsEnsembleId;
        this.sortingOrder = sortingOrder;
        this.aggregationSettings = aggregationSettings;
    }

    public static ResourceTypeCreateDto from(ResourceTypeDto resourceType) {
        return new ResourceTypeCreateDto(
                resourceType.getKey(),
                resourceType.getNameEn(),
                resourceType.getNameRu(),
                resourceType.getDescriptionEn(),
                resourceType.getDescriptionRu(),
                resourceType.getUnitsEnsembleId(),
                resourceType.getSortingOrder(),
                AggregationSettingsHelper.toAggregationSettingsInputDto(resourceType.getAggregationSettings()
                        .orElse(null))
        );
    }

    @Schema(description = "Key, unique per provider.", required = true)
    public Optional<String> getKey() {
        return Optional.ofNullable(key);
    }

    @Schema(description = "Name, english.", required = true)
    public Optional<String> getNameEn() {
        return Optional.ofNullable(nameEn);
    }

    @Schema(description = "Name, russian.", required = true)
    public Optional<String> getNameRu() {
        return Optional.ofNullable(nameRu);
    }

    @Schema(description = "Description, english.", required = true)
    public Optional<String> getDescriptionEn() {
        return Optional.ofNullable(descriptionEn);
    }

    @Schema(description = "Description, russian.", required = true)
    public Optional<String> getDescriptionRu() {
        return Optional.ofNullable(descriptionRu);
    }

    @Schema(description = "Id of units ensemble for all resources of this type.", required = true)
    public Optional<String> getUnitsEnsembleId() {
        return Optional.ofNullable(unitsEnsembleId);
    }

    @Schema(description = "Sorting order.")
    public Optional<Long> getSortingOrder() {
        return Optional.ofNullable(sortingOrder);
    }

    @Schema(description = "Aggregation settings.")
    public Optional<AggregationSettingsInputDto> getAggregationSettings() {
        return Optional.ofNullable(aggregationSettings);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        ResourceTypeCreateDto that = (ResourceTypeCreateDto) o;
        return Objects.equals(key, that.key) &&
                Objects.equals(nameEn, that.nameEn) &&
                Objects.equals(nameRu, that.nameRu) &&
                Objects.equals(descriptionEn, that.descriptionEn) &&
                Objects.equals(descriptionRu, that.descriptionRu) &&
                Objects.equals(unitsEnsembleId, that.unitsEnsembleId) &&
                Objects.equals(sortingOrder, that.sortingOrder) &&
                Objects.equals(aggregationSettings, that.aggregationSettings);
    }

    @Override
    public int hashCode() {
        return Objects.hash(key, nameEn, nameRu, descriptionEn, descriptionRu, unitsEnsembleId, sortingOrder,
                aggregationSettings);
    }

    @Override
    public String toString() {
        return "ResourceTypeCreateDto{" +
                "key='" + key + '\'' +
                ", nameEn='" + nameEn + '\'' +
                ", nameRu='" + nameRu + '\'' +
                ", descriptionEn='" + descriptionEn + '\'' +
                ", descriptionRu='" + descriptionRu + '\'' +
                ", unitsEnsembleId='" + unitsEnsembleId + '\'' +
                ", sortingOrder=" + sortingOrder +
                ", aggregationSettings=" + aggregationSettings +
                '}';
    }
}
