package ru.yandex.intranet.d.web.model.resources.directory.types;

import java.util.Objects;
import java.util.Optional;

import javax.annotation.Nullable;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import io.swagger.v3.oas.annotations.media.Schema;

import ru.yandex.intranet.d.model.resources.types.ResourceTypeModel;
import ru.yandex.intranet.d.util.AggregationSettingsHelper;
import ru.yandex.intranet.d.web.model.providers.AggregationSettingsDto;

/**
 * Resource type DTO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Schema(description = "Resource type.")
@JsonIgnoreProperties(ignoreUnknown = true)
public final class ResourceTypeDto {

    private final String id;
    private final String providerId;
    private final long version;
    private final String key;
    private final String nameEn;
    private final String nameRu;
    private final String descriptionEn;
    private final String descriptionRu;
    private final String unitsEnsembleId;
    private final @Nullable Long sortingOrder;
    private final @Nullable AggregationSettingsDto aggregationSettings;

    @JsonCreator
    @SuppressWarnings("ParameterNumber")
    public ResourceTypeDto(@JsonProperty("id") String id,
                           @JsonProperty("providerId") String providerId,
                           @JsonProperty("version") long version,
                           @JsonProperty("key") String key,
                           @JsonProperty("nameEn") String nameEn,
                           @JsonProperty("nameRu") String nameRu,
                           @JsonProperty("descriptionEn") String descriptionEn,
                           @JsonProperty("descriptionRu") String descriptionRu,
                           String unitsEnsembleId,
                           @JsonProperty("sortingOrder") @Nullable Long sortingOrder,
                           @JsonProperty("aggregationSettings") @Nullable AggregationSettingsDto aggregationSettings) {
        this.id = id;
        this.providerId = providerId;
        this.version = version;
        this.key = key;
        this.nameEn = nameEn;
        this.nameRu = nameRu;
        this.descriptionEn = descriptionEn;
        this.descriptionRu = descriptionRu;
        this.unitsEnsembleId = unitsEnsembleId;
        this.sortingOrder = sortingOrder;
        this.aggregationSettings = aggregationSettings;
    }

    public static ResourceTypeDto from(ResourceTypeModel resourceType) {
        return new ResourceTypeDto(
                resourceType.getId(),
                resourceType.getProviderId(),
                resourceType.getVersion(),
                resourceType.getKey(),
                resourceType.getNameEn(),
                resourceType.getNameRu(),
                resourceType.getDescriptionEn(),
                resourceType.getDescriptionRu(),
                resourceType.getUnitsEnsembleId(),
                resourceType.getSortingOrder(),
                AggregationSettingsHelper.toAggregationSettingsDto(resourceType.getAggregationSettings().orElse(null))
        );
    }

    @Schema(description = "Unique id.", required = true)
    public String getId() {
        return id;
    }

    @Schema(description = "Provider id.", required = true)
    public String getProviderId() {
        return providerId;
    }

    @Schema(description = "Version.", required = true)
    public long getVersion() {
        return version;
    }

    @Schema(description = "Key, unique per provider.", required = true)
    public String getKey() {
        return key;
    }

    @Schema(description = "Name, english.", required = true)
    public String getNameEn() {
        return nameEn;
    }

    @Schema(description = "Name, russian.", required = true)
    public String getNameRu() {
        return nameRu;
    }

    @Schema(description = "Description, english.", required = true)
    public String getDescriptionEn() {
        return descriptionEn;
    }

    @Schema(description = "Description, russian.", required = true)
    public String getDescriptionRu() {
        return descriptionRu;
    }

    @Schema(description = "Id of units ensemble for all resources of this type.", required = true)
    public String getUnitsEnsembleId() {
        return unitsEnsembleId;
    }

    @Nullable
    @Schema(description = "Sorting order.")
    public Long getSortingOrder() {
        return sortingOrder;
    }

    @Schema(description = "Aggregation settings.")
    public Optional<AggregationSettingsDto> getAggregationSettings() {
        return Optional.ofNullable(aggregationSettings);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        ResourceTypeDto that = (ResourceTypeDto) o;
        return version == that.version &&
                Objects.equals(id, that.id) &&
                Objects.equals(providerId, that.providerId) &&
                Objects.equals(key, that.key) &&
                Objects.equals(nameEn, that.nameEn) &&
                Objects.equals(nameRu, that.nameRu) &&
                Objects.equals(descriptionEn, that.descriptionEn) &&
                Objects.equals(descriptionRu, that.descriptionRu) &&
                Objects.equals(unitsEnsembleId, that.unitsEnsembleId) &&
                Objects.equals(sortingOrder, that.sortingOrder) &&
                Objects.equals(aggregationSettings, that.aggregationSettings);
    }

    @Override
    public int hashCode() {
        return Objects.hash(id, providerId, version, key, nameEn, nameRu, descriptionEn, descriptionRu,
                unitsEnsembleId, sortingOrder, aggregationSettings);
    }

    @Override
    public String toString() {
        return "ResourceTypeDto{" +
                "id='" + id + '\'' +
                ", providerId='" + providerId + '\'' +
                ", version=" + version +
                ", key='" + key + '\'' +
                ", nameEn='" + nameEn + '\'' +
                ", nameRu='" + nameRu + '\'' +
                ", descriptionEn='" + descriptionEn + '\'' +
                ", descriptionRu='" + descriptionRu + '\'' +
                ", unitsEnsembleId='" + unitsEnsembleId + '\'' +
                ", sortingOrder=" + sortingOrder +
                ", aggregationSettings=" + aggregationSettings +
                '}';
    }

}
