package ru.yandex.intranet.d.web.model.transfers.api;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import io.swagger.v3.oas.annotations.media.Schema;

/**
 * Reverse transfer parameters DTO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Schema(description = "Reserve transfer request.")
@JsonIgnoreProperties(ignoreUnknown = true)
public final class CreateReserveTransferDto {

    private final String providerId;
    private final String folderId;
    private final List<CreateReserveResourceTransferDto> resourceTransfers;

    @JsonCreator
    public CreateReserveTransferDto(String providerId,
                                    String folderId,
                                    List<CreateReserveResourceTransferDto> resourceTransfers) {
        this.providerId = providerId;
        this.folderId = folderId;
        this.resourceTransfers = resourceTransfers;
    }

    public static Builder builder() {
        return new Builder();
    }

    @Schema(description = "Provider id.", required = true)
    public Optional<String> getProviderId() {
        return Optional.ofNullable(providerId);
    }

    @Schema(description = "Folder id.", required = true)
    public Optional<String> getFolderId() {
        return Optional.ofNullable(folderId);
    }

    @Schema(description = "Reserve quota resource transfers.", required = true)
    public Optional<List<CreateReserveResourceTransferDto>> getResourceTransfers() {
        return Optional.ofNullable(resourceTransfers);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        CreateReserveTransferDto that = (CreateReserveTransferDto) o;
        return Objects.equals(providerId, that.providerId) &&
                Objects.equals(folderId, that.folderId) &&
                Objects.equals(resourceTransfers, that.resourceTransfers);
    }

    @Override
    public int hashCode() {
        return Objects.hash(providerId, folderId, resourceTransfers);
    }

    @Override
    public String toString() {
        return "CreateReserveTransferDto{" +
                "providerId='" + providerId + '\'' +
                ", folderId='" + folderId + '\'' +
                ", resourceTransfers=" + resourceTransfers +
                '}';
    }

    public static final class Builder {

        private final List<CreateReserveResourceTransferDto> resourceTransfers = new ArrayList<>();

        private String providerId;
        private String folderId;

        private Builder() {
        }

        public Builder providerId(String providerId) {
            this.providerId = providerId;
            return this;
        }

        public Builder folderId(String folderId) {
            this.folderId = folderId;
            return this;
        }

        public Builder addResourceTransfer(CreateReserveResourceTransferDto resourceTransfer) {
            this.resourceTransfers.add(resourceTransfer);
            return this;
        }

        public Builder addResourceTransfers(CreateReserveResourceTransferDto... resourceTransfers) {
            this.resourceTransfers.addAll(Arrays.asList(resourceTransfers));
            return this;
        }

        public Builder addResourceTransfers(Collection<? extends CreateReserveResourceTransferDto> resourceTransfers) {
            this.resourceTransfers.addAll(resourceTransfers);
            return this;
        }

        public CreateReserveTransferDto build() {
            return new CreateReserveTransferDto(providerId, folderId, resourceTransfers);
        }

    }

}
