package ru.yandex.intranet.d.web.model.transfers.api;

import java.util.Objects;
import java.util.Optional;

import javax.annotation.Nullable;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import io.swagger.v3.oas.annotations.media.Schema;

import ru.yandex.intranet.d.web.model.transfers.TransferRequestTypeDto;

/**
 * Create transfer request DTO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Schema(description = "Transfer request.")
@JsonIgnoreProperties(ignoreUnknown = true)
public final class CreateTransferRequestDto {

    private final TransferRequestTypeDto requestType;
    private final String description;
    private final Boolean addConfirmation;
    private final CreateTransferRequestParametersDto parameters;
    @Nullable
    private final TransferLoanParametersDto loanParameters;

    @JsonCreator
    public CreateTransferRequestDto(TransferRequestTypeDto requestType,
                                    String description,
                                    Boolean addConfirmation,
                                    CreateTransferRequestParametersDto parameters,
                                    @Nullable TransferLoanParametersDto loanParameters) {
        this.requestType = requestType;
        this.description = description;
        this.addConfirmation = addConfirmation;
        this.parameters = parameters;
        this.loanParameters = loanParameters;
    }

    public static Builder builder() {
        return new Builder();
    }

    @Schema(description = "Transfer request type.", required = true)
    public Optional<TransferRequestTypeDto> getRequestType() {
        return Optional.ofNullable(requestType);
    }

    @Schema(description = "Transfer request description.")
    public Optional<String> getDescription() {
        return Optional.ofNullable(description);
    }

    @Schema(description = "'Transfer request is auto-confirmed by author with enough permissions' flag.",
            required = true)
    public Optional<Boolean> getAddConfirmation() {
        return Optional.ofNullable(addConfirmation);
    }

    @Schema(description = "Transfer request parameters.", required = true)
    public Optional<CreateTransferRequestParametersDto> getParameters() {
        return Optional.ofNullable(parameters);
    }

    @Schema(description = "Loan parameters.")
    public Optional<TransferLoanParametersDto> getLoanParameters() {
        return Optional.ofNullable(loanParameters);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        CreateTransferRequestDto that = (CreateTransferRequestDto) o;
        return requestType == that.requestType &&
                Objects.equals(description, that.description) &&
                Objects.equals(addConfirmation, that.addConfirmation) &&
                Objects.equals(parameters, that.parameters) &&
                Objects.equals(loanParameters, that.loanParameters);
    }

    @Override
    public int hashCode() {
        return Objects.hash(requestType, description, addConfirmation, parameters, loanParameters);
    }

    @Override
    public String toString() {
        return "CreateTransferRequestDto{" +
                "requestType=" + requestType +
                ", description='" + description + '\'' +
                ", addConfirmation=" + addConfirmation +
                ", parameters=" + parameters +
                ", loanParameters=" + loanParameters +
                '}';
    }

    public static final class Builder {

        private TransferRequestTypeDto requestType;
        private String description;
        private Boolean addConfirmation;
        private CreateTransferRequestParametersDto parameters;
        @Nullable
        private TransferLoanParametersDto loanParameters;

        private Builder() {
        }

        public Builder requestType(TransferRequestTypeDto requestType) {
            this.requestType = requestType;
            return this;
        }

        public Builder description(String description) {
            this.description = description;
            return this;
        }

        public Builder addConfirmation(Boolean addConfirmation) {
            this.addConfirmation = addConfirmation;
            return this;
        }

        public Builder parameters(CreateTransferRequestParametersDto parameters) {
            this.parameters = parameters;
            return this;
        }

        public Builder loanParameters(@Nullable TransferLoanParametersDto loanParameters) {
            this.loanParameters = loanParameters;
            return this;
        }

        public CreateTransferRequestDto build() {
            return new CreateTransferRequestDto(requestType, description, addConfirmation, parameters, loanParameters);
        }

    }

}
