package ru.yandex.intranet.d.web.model.transfers.api;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import io.swagger.v3.oas.annotations.media.Schema;

/**
 * Create transfer request parameters DTO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Schema(description = "Transfer request parameters.")
@JsonIgnoreProperties(ignoreUnknown = true)
public final class CreateTransferRequestParametersDto {

    private final List<CreateQuotaTransferDto> quotaTransfers;
    private final CreateReserveTransferDto reserveTransfer;
    private final List<CreateProvisionTransferDto> provisionTransfers;
    private final Boolean provideOverCommitReserve;

    @JsonCreator
    public CreateTransferRequestParametersDto(List<CreateQuotaTransferDto> quotaTransfers,
                                              CreateReserveTransferDto reserveTransfer,
                                              List<CreateProvisionTransferDto> provisionTransfers,
                                              Boolean provideOverCommitReserve) {
        this.quotaTransfers = quotaTransfers;
        this.reserveTransfer = reserveTransfer;
        this.provisionTransfers = provisionTransfers;
        this.provideOverCommitReserve = provideOverCommitReserve;
    }

    public static Builder builder() {
        return new Builder();
    }

    @Schema(description = "Quota transfer requests.")
    public Optional<List<CreateQuotaTransferDto>> getQuotaTransfers() {
        return Optional.ofNullable(quotaTransfers);
    }

    @Schema(description = "Reserve transfer requests.")
    public Optional<CreateReserveTransferDto> getReserveTransfer() {
        return Optional.ofNullable(reserveTransfer);
    }

    @Schema(description = "Provision transfer requests.")
    public Optional<List<CreateProvisionTransferDto>> getProvisionTransfers() {
        return Optional.ofNullable(provisionTransfers);
    }

    @Schema(description = "Provide over commit reserve.")
    public Optional<Boolean> getProvideOverCommitReserve() {
        return Optional.ofNullable(provideOverCommitReserve);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        CreateTransferRequestParametersDto that = (CreateTransferRequestParametersDto) o;
        return Objects.equals(quotaTransfers, that.quotaTransfers) &&
                Objects.equals(reserveTransfer, that.reserveTransfer) &&
                Objects.equals(provisionTransfers, that.provisionTransfers) &&
                Objects.equals(provideOverCommitReserve, that.provideOverCommitReserve);
    }

    @Override
    public int hashCode() {
        return Objects.hash(quotaTransfers, reserveTransfer, provisionTransfers, provideOverCommitReserve);
    }

    @Override
    public String toString() {
        return "CreateTransferRequestParametersDto{" +
                "quotaTransfers=" + quotaTransfers +
                ", reserveTransfer=" + reserveTransfer +
                ", provisionTransfers=" + provisionTransfers +
                ", provideOverCommitReserve=" + provideOverCommitReserve +
                '}';
    }

    public static final class Builder {

        private final List<CreateQuotaTransferDto> quotaTransfers = new ArrayList<>();
        private final List<CreateProvisionTransferDto> provisionTransfers = new ArrayList<>();
        private CreateReserveTransferDto reserveTransfer;
        private Boolean provideOverCommitReserve;


        private Builder() {
        }

        public Builder addQuotaTransfer(CreateQuotaTransferDto quotaTransfer) {
            this.quotaTransfers.add(quotaTransfer);
            return this;
        }

        public Builder addQuotaTransfers(CreateQuotaTransferDto... quotaTransfers) {
            this.quotaTransfers.addAll(Arrays.asList(quotaTransfers));
            return this;
        }

        public Builder addQuotaTransfers(Collection<? extends CreateQuotaTransferDto> quotaTransfers) {
            this.quotaTransfers.addAll(quotaTransfers);
            return this;
        }

        public Builder reserveTransfer(CreateReserveTransferDto reserveTransfer) {
            this.reserveTransfer = reserveTransfer;
            return this;
        }

        public Builder addProvisionTransfer(CreateProvisionTransferDto provisionTransfer) {
            this.provisionTransfers.add(provisionTransfer);
            return this;
        }

        public Builder addProvisionTransfers(List<CreateProvisionTransferDto> provisionTransfers) {
            this.provisionTransfers.addAll(provisionTransfers);
            return this;
        }

        public Builder provideOverCommitReserve(Boolean provideOverCommitReserve) {
            this.provideOverCommitReserve = provideOverCommitReserve;
            return this;
        }

        public CreateTransferRequestParametersDto build() {
            return new CreateTransferRequestParametersDto(quotaTransfers, reserveTransfer, provisionTransfers,
                    provideOverCommitReserve);
        }

    }

}
