package ru.yandex.intranet.d.web.model.transfers.api;

import java.util.Objects;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import io.swagger.v3.oas.annotations.media.Schema;

/**
 * Quota resource transfer DTO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Schema(description = "Quota resource transfer request.")
@JsonIgnoreProperties(ignoreUnknown = true)
public final class QuotaResourceTransferDto {

    private final String providerId;
    private final String resourceId;
    private final long delta;
    private final String deltaUnitKey;

    @JsonCreator
    public QuotaResourceTransferDto(String providerId, String resourceId, long delta, String deltaUnitKey) {
        this.providerId = providerId;
        this.resourceId = resourceId;
        this.delta = delta;
        this.deltaUnitKey = deltaUnitKey;
    }

    public static Builder builder() {
        return new Builder();
    }

    @Schema(description = "Provider id.", required = true)
    public String getProviderId() {
        return providerId;
    }

    @Schema(description = "Resource id.", required = true)
    public String getResourceId() {
        return resourceId;
    }

    @Schema(description = "Transferred resource amount.", required = true)
    public long getDelta() {
        return delta;
    }

    @Schema(description = "Transferred resource amount unit key.", required = true)
    public String getDeltaUnitKey() {
        return deltaUnitKey;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        QuotaResourceTransferDto that = (QuotaResourceTransferDto) o;
        return delta == that.delta &&
                Objects.equals(providerId, that.providerId) &&
                Objects.equals(resourceId, that.resourceId) &&
                Objects.equals(deltaUnitKey, that.deltaUnitKey);
    }

    @Override
    public int hashCode() {
        return Objects.hash(providerId, resourceId, delta, deltaUnitKey);
    }

    @Override
    public String toString() {
        return "QuotaResourceTransferDto{" +
                "providerId='" + providerId + '\'' +
                ", resourceId='" + resourceId + '\'' +
                ", delta=" + delta +
                ", deltaUnitKey='" + deltaUnitKey + '\'' +
                '}';
    }

    public static final class Builder {

        private String providerId;
        private String resourceId;
        private Long delta;
        private String deltaUnitKey;

        private Builder() {
        }

        public Builder providerId(String providerId) {
            this.providerId = providerId;
            return this;
        }

        public Builder resourceId(String resourceId) {
            this.resourceId = resourceId;
            return this;
        }

        public Builder delta(long delta) {
            this.delta = delta;
            return this;
        }

        public Builder deltaUnitKey(String deltaUnitKey) {
            this.deltaUnitKey = deltaUnitKey;
            return this;
        }

        public QuotaResourceTransferDto build() {
            Objects.requireNonNull(providerId, "ProviderId is required");
            Objects.requireNonNull(resourceId, "ResourceId is required");
            Objects.requireNonNull(delta, "Delta is required");
            Objects.requireNonNull(deltaUnitKey, "DeltaUnitKey is required");
            return new QuotaResourceTransferDto(providerId, resourceId, delta, deltaUnitKey);
        }

    }

}
