package ru.yandex.intranet.d.web.model.transfers.api;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import io.swagger.v3.oas.annotations.media.Schema;

/**
 * Quota transfer parameters DTO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Schema(description = "Quota transfer request.")
@JsonIgnoreProperties(ignoreUnknown = true)
public final class QuotaTransferDto {

    private final String folderId;
    private final List<QuotaResourceTransferDto> resourceTransfers;

    @JsonCreator
    public QuotaTransferDto(String folderId, List<QuotaResourceTransferDto> resourceTransfers) {
        this.folderId = folderId;
        this.resourceTransfers = resourceTransfers;
    }

    public static Builder builder() {
        return new Builder();
    }

    @Schema(description = "Folder id.", required = true)
    public String getFolderId() {
        return folderId;
    }

    @Schema(description = "Quota resource transfers.", required = true)
    public List<QuotaResourceTransferDto> getResourceTransfers() {
        return resourceTransfers;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        QuotaTransferDto that = (QuotaTransferDto) o;
        return Objects.equals(folderId, that.folderId) &&
                Objects.equals(resourceTransfers, that.resourceTransfers);
    }

    @Override
    public int hashCode() {
        return Objects.hash(folderId, resourceTransfers);
    }

    @Override
    public String toString() {
        return "QuotaTransferDto{" +
                "folderId='" + folderId + '\'' +
                ", resourceTransfers=" + resourceTransfers +
                '}';
    }

    public static final class Builder {

        private final List<QuotaResourceTransferDto> resourceTransfers = new ArrayList<>();

        private String folderId;

        private Builder() {
        }

        public Builder folderId(String folderId) {
            this.folderId = folderId;
            return this;
        }

        public Builder addResourceTransfer(QuotaResourceTransferDto resourceTransfer) {
            this.resourceTransfers.add(resourceTransfer);
            return this;
        }

        public Builder addResourceTransfers(QuotaResourceTransferDto... resourceTransfers) {
            this.resourceTransfers.addAll(Arrays.asList(resourceTransfers));
            return this;
        }

        public Builder addResourceTransfers(Collection<? extends QuotaResourceTransferDto> resourceTransfers) {
            this.resourceTransfers.addAll(resourceTransfers);
            return this;
        }

        public QuotaTransferDto build() {
            Objects.requireNonNull(folderId, "FolderId is required");
            return new QuotaTransferDto(folderId, resourceTransfers);
        }

    }

}
