package ru.yandex.intranet.d.web.model.transfers.api;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import io.swagger.v3.oas.annotations.media.Schema;

/**
 * Reserve quota transfer parameters DTO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Schema(description = "Reserve quota transfer request.")
@JsonIgnoreProperties(ignoreUnknown = true)
public final class ReserveTransferDto {

    private final String providerId;
    private final String folderId;
    private final List<ReserveResourceTransferDto> resourceTransfers;

    @JsonCreator
    public ReserveTransferDto(String providerId, String folderId, List<ReserveResourceTransferDto> resourceTransfers) {
        this.providerId = providerId;
        this.folderId = folderId;
        this.resourceTransfers = resourceTransfers;
    }

    public static Builder builder() {
        return new Builder();
    }

    @Schema(description = "Provider id.", required = true)
    public String getProviderId() {
        return providerId;
    }

    @Schema(description = "Folder id.", required = true)
    public String getFolderId() {
        return folderId;
    }

    @Schema(description = "Reserve quota resource transfers.", required = true)
    public List<ReserveResourceTransferDto> getResourceTransfers() {
        return resourceTransfers;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        ReserveTransferDto that = (ReserveTransferDto) o;
        return Objects.equals(providerId, that.providerId) &&
                Objects.equals(folderId, that.folderId) &&
                Objects.equals(resourceTransfers, that.resourceTransfers);
    }

    @Override
    public int hashCode() {
        return Objects.hash(providerId, folderId, resourceTransfers);
    }

    @Override
    public String toString() {
        return "ReserveTransferDto{" +
                "providerId='" + providerId + '\'' +
                ", folderId='" + folderId + '\'' +
                ", resourceTransfers=" + resourceTransfers +
                '}';
    }

    public static final class Builder {

        private final List<ReserveResourceTransferDto> resourceTransfers = new ArrayList<>();

        private String providerId;
        private String folderId;

        private Builder() {
        }

        public Builder providerId(String providerId) {
            this.providerId = providerId;
            return this;
        }

        public Builder folderId(String folderId) {
            this.folderId = folderId;
            return this;
        }

        public Builder addResourceTransfer(ReserveResourceTransferDto resourceTransfer) {
            this.resourceTransfers.add(resourceTransfer);
            return this;
        }

        public Builder addResourceTransfers(ReserveResourceTransferDto... resourceTransfers) {
            this.resourceTransfers.addAll(Arrays.asList(resourceTransfers));
            return this;
        }

        public Builder addResourceTransfers(Collection<? extends ReserveResourceTransferDto> resourceTransfers) {
            this.resourceTransfers.addAll(resourceTransfers);
            return this;
        }

        public ReserveTransferDto build() {
            Objects.requireNonNull(providerId, "ProviderId is required");
            Objects.requireNonNull(folderId, "FolderId is required");
            return new ReserveTransferDto(providerId, folderId, resourceTransfers);
        }

    }

}
