package ru.yandex.intranet.d.web.model.transfers.api;

import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import io.swagger.v3.oas.annotations.media.Schema;

/**
 * Transfer request application details DTO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Schema(description = "Transfer request application details.")
@JsonIgnoreProperties(ignoreUnknown = true)
public final class TransferRequestApplicationDetailsDto {

    private final TransferRequestApplicationFailuresDto failures;
    private final Map<String, TransferOperationStatusDto> operationStatusById;
    private final Map<String, TransferRequestApplicationFailuresDto> failuresByOperationId;

    @JsonCreator
    public TransferRequestApplicationDetailsDto(
            TransferRequestApplicationFailuresDto failures,
            Map<String, TransferOperationStatusDto> operationStatusById,
            Map<String, TransferRequestApplicationFailuresDto> failuresByOperationId
    ) {
        this.failures = failures;
        this.operationStatusById = Objects.requireNonNullElse(operationStatusById, Map.of());
        this.failuresByOperationId = Objects.requireNonNullElse(failuresByOperationId, Map.of());
    }

    public static Builder builder() {
        return new Builder();
    }

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    @Schema(description = "Transfer request application failures.")
    public Optional<TransferRequestApplicationFailuresDto> getFailures() {
        return Optional.ofNullable(failures);
    }

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    @Schema(description = "Operation status by operation id.")
    public Map<String, TransferOperationStatusDto> getOperationStatusById() {
        return operationStatusById;
    }

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    @Schema(description = "Operation failures by operation id.")
    public Map<String, TransferRequestApplicationFailuresDto> getFailuresByOperationId() {
        return failuresByOperationId;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        TransferRequestApplicationDetailsDto that = (TransferRequestApplicationDetailsDto) o;
        return Objects.equals(failures, that.failures) &&
                Objects.equals(failuresByOperationId, that.failuresByOperationId) &&
                Objects.equals(operationStatusById, that.operationStatusById);
    }

    @Override
    public int hashCode() {
        return Objects.hash(failures, operationStatusById, failuresByOperationId);
    }

    @Override
    public String toString() {
        return "TransferRequestApplicationDetailsDto{" +
                "failures=" + failures +
                ", operationStatusById=" + operationStatusById +
                ", failuresByOperationId=" + failuresByOperationId +
                '}';
    }

    public static final class Builder {
        private final Map<String, TransferOperationStatusDto> operationStatusById = new HashMap<>();
        private final Map<String, TransferRequestApplicationFailuresDto> failuresByOperationId = new HashMap<>();

        private TransferRequestApplicationFailuresDto failures;

        private Builder() {
        }

        public Builder failures(TransferRequestApplicationFailuresDto failures) {
            this.failures = failures;
            return this;
        }

        public Builder failuresByOperationId(
                Map<String, TransferRequestApplicationFailuresDto> failuresByOperationId
        ) {
            this.failuresByOperationId.putAll(failuresByOperationId);
            return this;
        }

        public Builder operationStatusById(Map<String, TransferOperationStatusDto> operationStatusById) {
            this.operationStatusById.putAll(operationStatusById);
            return this;
        }

        public TransferRequestApplicationDetailsDto build() {
            return new TransferRequestApplicationDetailsDto(failures, operationStatusById, failuresByOperationId);
        }

    }

}
