package ru.yandex.intranet.d.web.model.transfers.api;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import io.swagger.v3.oas.annotations.media.Schema;

/**
 * Transfer request application failures DTO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Schema(description = "Transfer request application failures.")
@JsonIgnoreProperties(ignoreUnknown = true)
public final class TransferRequestApplicationFailuresDto {

    private final List<String> errors;
    private final Map<String, List<String>> fieldErrors;
    private final Map<String, List<Object>> details;

    @JsonCreator
    public TransferRequestApplicationFailuresDto(List<String> errors,
                                                 Map<String, List<String>> fieldErrors,
                                                 Map<String, List<Object>> details) {
        this.errors = errors != null ? errors : List.of();
        this.fieldErrors = fieldErrors != null ? fieldErrors : Map.of();
        this.details = details != null ? details : Map.of();
    }

    public static Builder builder() {
        return new Builder();
    }

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    @Schema(description = "Errors list.")
    public List<String> getErrors() {
        return errors;
    }

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    @Schema(description = "Errors per field.")
    public Map<String, List<String>> getFieldErrors() {
        return fieldErrors;
    }

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    @Schema(description = "Error details.")
    public Map<String, List<Object>> getDetails() {
        return details;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        TransferRequestApplicationFailuresDto that = (TransferRequestApplicationFailuresDto) o;
        return Objects.equals(errors, that.errors) &&
                Objects.equals(fieldErrors, that.fieldErrors) &&
                Objects.equals(details, that.details);
    }

    @Override
    public int hashCode() {
        return Objects.hash(errors, fieldErrors, details);
    }

    @Override
    public String toString() {
        return "TransferRequestApplicationFailuresDto{" +
                "errors=" + errors +
                ", fieldErrors=" + fieldErrors +
                ", details=" + details +
                '}';
    }

    public static final class Builder {

        private final List<String> errors = new ArrayList<>();
        private final Map<String, List<String>> fieldErrors = new HashMap<>();
        private final Map<String, List<Object>> details = new HashMap<>();

        private Builder() {
        }

        public Builder addError(String value) {
            this.errors.add(value);
            return this;
        }

        public Builder addErrors(String... values) {
            this.errors.addAll(Arrays.asList(values));
            return this;
        }

        public Builder addErrors(Collection<? extends String> values) {
            this.errors.addAll(values);
            return this;
        }

        public Builder addFieldError(String field, String value) {
            this.fieldErrors.computeIfAbsent(field, k -> new ArrayList<>()).add(value);
            return this;
        }

        public Builder addFieldErrors(String field, String... values) {
            this.fieldErrors.computeIfAbsent(field, k -> new ArrayList<>()).addAll(Arrays.asList(values));
            return this;
        }

        public Builder addFieldErrors(String field, Collection<? extends String> values) {
            this.fieldErrors.computeIfAbsent(field, k -> new ArrayList<>()).addAll(values);
            return this;
        }

        public Builder addFieldErrors(Map<String, ? extends Collection<? extends String>> map) {
            map.forEach((folderId, opLogIds) -> this.fieldErrors.computeIfAbsent(folderId, k -> new ArrayList<>())
                    .addAll(opLogIds));
            return this;
        }

        public Builder addDetails(Map<String, ? extends Collection<?>> map) {
            map.forEach((detail, values) -> this.details.computeIfAbsent(detail, k -> new ArrayList<>())
                    .addAll(values));
            return this;
        }

        public TransferRequestApplicationFailuresDto build() {
            return new TransferRequestApplicationFailuresDto(errors, fieldErrors, details);
        }

    }

}
