package ru.yandex.intranet.d.web.model.transfers.api;

import java.time.Instant;
import java.util.Objects;
import java.util.Optional;

import javax.annotation.Nullable;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonFormat;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import io.swagger.v3.oas.annotations.media.Schema;

import ru.yandex.intranet.d.web.model.transfers.TransferRequestStatusDto;
import ru.yandex.intranet.d.web.model.transfers.TransferRequestSubtypeDto;
import ru.yandex.intranet.d.web.model.transfers.TransferRequestTypeDto;

/**
 * Transfer request DTO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Schema(description = "Transfer request.")
@JsonIgnoreProperties(ignoreUnknown = true)
public final class TransferRequestDto {

    private final String id;
    private final long version;
    private final String description;
    private final String trackerIssueKey;
    private final TransferRequestTypeDto requestType;
    private final TransferRequestStatusDto status;
    private final String createdBy;
    private final String updatedBy;
    private final Instant createdAt;
    private final Instant updatedAt;
    private final Instant appliedAt;
    private final TransferRequestParametersDto parameters;
    private final TransferRequestResponsibleDto transferResponsible;
    private final TransferRequestVotesDto votes;
    private final TransferRequestApplicationDetailsDto application;
    private final TransferLoanMetaDto loanMeta;
    private final TransferRequestSubtypeDto requestSubtype;

    @JsonCreator
    @SuppressWarnings("ParameterNumber")
    public TransferRequestDto(
            String id,
            long version,
            String description,
            String trackerIssueKey,
            TransferRequestTypeDto requestType,
            TransferRequestStatusDto status,
            String createdBy,
            String updatedBy,
            @JsonFormat(shape = JsonFormat.Shape.STRING, pattern = "yyyy-MM-dd'T'HH:mm:ss.SSS'Z'", timezone = "UTC")
            Instant createdAt,
            @JsonFormat(shape = JsonFormat.Shape.STRING, pattern = "yyyy-MM-dd'T'HH:mm:ss.SSS'Z'", timezone = "UTC")
            Instant updatedAt,
            @JsonFormat(shape = JsonFormat.Shape.STRING, pattern = "yyyy-MM-dd'T'HH:mm:ss.SSS'Z'", timezone = "UTC")
            Instant appliedAt,
            TransferRequestParametersDto parameters,
            TransferRequestResponsibleDto transferResponsible,
            TransferRequestVotesDto votes,
            TransferRequestApplicationDetailsDto application,
            TransferLoanMetaDto loanMeta,
            TransferRequestSubtypeDto requestSubtype) {
        this.id = id;
        this.version = version;
        this.description = description;
        this.trackerIssueKey = trackerIssueKey;
        this.requestType = requestType;
        this.status = status;
        this.createdBy = createdBy;
        this.updatedBy = updatedBy;
        this.createdAt = createdAt;
        this.updatedAt = updatedAt;
        this.appliedAt = appliedAt;
        this.parameters = parameters;
        this.transferResponsible = transferResponsible;
        this.votes = votes;
        this.application = application;
        this.loanMeta = loanMeta;
        this.requestSubtype = requestSubtype;
    }

    public static Builder builder() {
        return new Builder();
    }

    @Schema(description = "Transfer request id.", required = true)
    public String getId() {
        return id;
    }

    @Schema(description = "Transfer request version.", required = true)
    public long getVersion() {
        return version;
    }

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    @Schema(description = "Transfer request description.")
    public Optional<String> getDescription() {
        return Optional.ofNullable(description);
    }

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    @Schema(description = "Transfer request Tracker issue key.")
    public Optional<String> getTrackerIssueKey() {
        return Optional.ofNullable(trackerIssueKey);
    }

    @Schema(description = "Transfer request type.", required = true)
    public TransferRequestTypeDto getRequestType() {
        return requestType;
    }

    @Schema(description = "Transfer request subtype.", required = true)
    public TransferRequestSubtypeDto getRequestSubtype() {
        return requestSubtype;
    }

    @Schema(description = "Transfer request status.", required = true)
    public TransferRequestStatusDto getStatus() {
        return status;
    }

    @Schema(description = "Transfer request author uid.", required = true)
    public String getCreatedBy() {
        return createdBy;
    }

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    @Schema(description = "Transfer request last editor uid.")
    public Optional<String> getUpdatedBy() {
        return Optional.ofNullable(updatedBy);
    }

    @JsonFormat(shape = JsonFormat.Shape.STRING, pattern = "yyyy-MM-dd'T'HH:mm:ss.SSS'Z'", timezone = "UTC")
    @Schema(description = "Transfer request creation timestamp.", required = true)
    public Instant getCreatedAt() {
        return createdAt;
    }

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    @JsonFormat(shape = JsonFormat.Shape.STRING, pattern = "yyyy-MM-dd'T'HH:mm:ss.SSS'Z'", timezone = "UTC")
    @Schema(description = "Transfer request last update timestamp.")
    public Optional<Instant> getUpdatedAt() {
        return Optional.ofNullable(updatedAt);
    }

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    @JsonFormat(shape = JsonFormat.Shape.STRING, pattern = "yyyy-MM-dd'T'HH:mm:ss.SSS'Z'", timezone = "UTC")
    @Schema(description = "Transfer request application timestamp.")
    public Optional<Instant> getAppliedAt() {
        return Optional.ofNullable(appliedAt);
    }

    @Schema(description = "Transfer request parameters.", required = true)
    public TransferRequestParametersDto getParameters() {
        return parameters;
    }

    @Schema(description = "Transfer request responsible.", required = true)
    public TransferRequestResponsibleDto getTransferResponsible() {
        return transferResponsible;
    }

    @Schema(description = "Transfer request votes.", required = true)
    public TransferRequestVotesDto getVotes() {
        return votes;
    }

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    @Schema(description = "Transfer request application details.")
    public Optional<TransferRequestApplicationDetailsDto> getApplication() {
        return Optional.ofNullable(application);
    }

    @Schema(description = "Loan metadata.")
    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    public Optional<TransferLoanMetaDto> getLoanMeta() {
        return Optional.ofNullable(loanMeta);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        TransferRequestDto that = (TransferRequestDto) o;
        return version == that.version &&
                Objects.equals(id, that.id) &&
                Objects.equals(description, that.description) &&
                Objects.equals(trackerIssueKey, that.trackerIssueKey) &&
                requestType == that.requestType &&
                requestSubtype == that.requestSubtype &&
                status == that.status &&
                Objects.equals(createdBy, that.createdBy) &&
                Objects.equals(updatedBy, that.updatedBy) &&
                Objects.equals(createdAt, that.createdAt) &&
                Objects.equals(updatedAt, that.updatedAt) &&
                Objects.equals(appliedAt, that.appliedAt) &&
                Objects.equals(parameters, that.parameters) &&
                Objects.equals(transferResponsible, that.transferResponsible) &&
                Objects.equals(votes, that.votes) &&
                Objects.equals(application, that.application) &&
                Objects.equals(loanMeta, that.loanMeta);
    }

    @Override
    public int hashCode() {
        return Objects.hash(id, version, description, trackerIssueKey, requestType, status, createdBy, updatedBy,
                createdAt, updatedAt, appliedAt, parameters, transferResponsible, votes, application, loanMeta,
                requestSubtype);
    }

    @Override
    public String toString() {
        return "TransferRequestDto{" +
                "id='" + id + '\'' +
                ", version=" + version +
                ", description='" + description + '\'' +
                ", trackerIssueKey='" + trackerIssueKey + '\'' +
                ", requestType=" + requestType +
                ", requestSubtype=" + requestSubtype +
                ", status=" + status +
                ", createdBy='" + createdBy + '\'' +
                ", updatedBy='" + updatedBy + '\'' +
                ", createdAt=" + createdAt +
                ", updatedAt=" + updatedAt +
                ", appliedAt=" + appliedAt +
                ", parameters=" + parameters +
                ", transferResponsible=" + transferResponsible +
                ", votes=" + votes +
                ", application=" + application +
                ", loanMeta=" + loanMeta +
                '}';
    }

    public static final class Builder {

        private String id;
        private Long version;
        private String description;
        private String trackerIssueKey;
        private TransferRequestTypeDto requestType;
        private TransferRequestStatusDto status;
        private String createdBy;
        private String updatedBy;
        private Instant createdAt;
        private Instant updatedAt;
        private Instant appliedAt;
        private TransferRequestParametersDto parameters;
        private TransferRequestResponsibleDto transferResponsible;
        private TransferRequestVotesDto votes;
        private TransferRequestApplicationDetailsDto application;
        @Nullable
        private TransferLoanMetaDto loanMeta;
        @Nullable
        private TransferRequestSubtypeDto requestSubtype;

        private Builder() {
        }

        public Builder id(String id) {
            this.id = id;
            return this;
        }

        public Builder version(long version) {
            this.version = version;
            return this;
        }

        public Builder description(String description) {
            this.description = description;
            return this;
        }

        public Builder trackerIssueKey(String trackerIssueKey) {
            this.trackerIssueKey = trackerIssueKey;
            return this;
        }

        public Builder requestType(TransferRequestTypeDto requestType) {
            this.requestType = requestType;
            return this;
        }

        public Builder requestSubtype(TransferRequestSubtypeDto requestSubtype) {
            this.requestSubtype = requestSubtype;
            return this;
        }

        public Builder status(TransferRequestStatusDto status) {
            this.status = status;
            return this;
        }

        public Builder createdBy(String createdBy) {
            this.createdBy = createdBy;
            return this;
        }

        public Builder updatedBy(String updatedBy) {
            this.updatedBy = updatedBy;
            return this;
        }

        public Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public Builder updatedAt(Instant updatedAt) {
            this.updatedAt = updatedAt;
            return this;
        }

        public Builder appliedAt(Instant appliedAt) {
            this.appliedAt = appliedAt;
            return this;
        }

        public Builder parameters(TransferRequestParametersDto parameters) {
            this.parameters = parameters;
            return this;
        }

        public Builder transferResponsible(TransferRequestResponsibleDto transferResponsible) {
            this.transferResponsible = transferResponsible;
            return this;
        }

        public Builder votes(TransferRequestVotesDto votes) {
            this.votes = votes;
            return this;
        }

        public Builder application(TransferRequestApplicationDetailsDto application) {
            this.application = application;
            return this;
        }

        public Builder loanMeta(@Nullable TransferLoanMetaDto loanMeta) {
            this.loanMeta = loanMeta;
            return this;
        }

        public TransferRequestDto build() {
            Objects.requireNonNull(id, "Id is required");
            Objects.requireNonNull(version, "Version is required");
            Objects.requireNonNull(requestType, "RequestType is required");
            Objects.requireNonNull(status, "Status is required");
            Objects.requireNonNull(createdBy, "CreatedBy is required");
            Objects.requireNonNull(createdAt, "CreatedAt is required");
            Objects.requireNonNull(parameters, "Parameters is required");
            Objects.requireNonNull(transferResponsible, "TransferResponsible is required");
            Objects.requireNonNull(votes, "TransferVotes is required");
            return new TransferRequestDto(id, version, description, trackerIssueKey, requestType, status,
                    createdBy, updatedBy, createdAt, updatedAt, appliedAt, parameters, transferResponsible,
                    votes, application, loanMeta, requestSubtype);
        }

    }

}
