package ru.yandex.intranet.d.web.model.transfers.api;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import io.swagger.v3.oas.annotations.media.Schema;

/**
 * Transfer request parameters DTO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Schema(description = "Transfer request parameters.")
@JsonIgnoreProperties(ignoreUnknown = true)
public final class TransferRequestParametersDto {

    private final List<QuotaTransferDto> quotaTransfers;
    private final ReserveTransferDto reserveTransfer;
    private final List<ProvisionTransferDto> provisionTransfers;

    @JsonCreator
    public TransferRequestParametersDto(List<QuotaTransferDto> quotaTransfers,
                                        ReserveTransferDto reserveTransfer,
                                        List<ProvisionTransferDto> provisionTransfers) {
        this.quotaTransfers = quotaTransfers != null ? quotaTransfers : List.of();
        this.reserveTransfer = reserveTransfer;
        this.provisionTransfers = provisionTransfers != null ? provisionTransfers : List.of();
    }

    public static Builder builder() {
        return new Builder();
    }

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    @Schema(description = "Quota transfer requests.")
    public List<QuotaTransferDto> getQuotaTransfers() {
        return quotaTransfers;
    }

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    @Schema(description = "Reserve quota transfer requests.")
    public Optional<ReserveTransferDto> getReserveTransfer() {
        return Optional.ofNullable(reserveTransfer);
    }

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    @Schema(description = "Provision quota transfer requests.")
    public List<ProvisionTransferDto> getProvisionTransfers() {
        return provisionTransfers;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        TransferRequestParametersDto that = (TransferRequestParametersDto) o;
        return Objects.equals(quotaTransfers, that.quotaTransfers) &&
                Objects.equals(reserveTransfer, that.reserveTransfer) &&
                Objects.equals(provisionTransfers, that.provisionTransfers);
    }

    @Override
    public int hashCode() {
        return Objects.hash(quotaTransfers, reserveTransfer, provisionTransfers);
    }

    @Override
    public String toString() {
        return "TransferRequestParametersDto{" +
                "quotaTransfers=" + quotaTransfers +
                ", reserveTransfer=" + reserveTransfer +
                ", provisionTransfers=" + provisionTransfers +
                '}';
    }

    public static final class Builder {

        private final List<QuotaTransferDto> quotaTransfers = new ArrayList<>();
        private final List<ProvisionTransferDto> provisionTransfers = new ArrayList<>();
        private ReserveTransferDto reserveTransfer;

        private Builder() {
        }

        public Builder addQuotaTransfer(QuotaTransferDto quotaTransfer) {
            this.quotaTransfers.add(quotaTransfer);
            return this;
        }

        public Builder addQuotaTransfers(QuotaTransferDto... quotaTransfers) {
            this.quotaTransfers.addAll(Arrays.asList(quotaTransfers));
            return this;
        }

        public Builder addQuotaTransfers(Collection<? extends QuotaTransferDto> quotaTransfers) {
            this.quotaTransfers.addAll(quotaTransfers);
            return this;
        }

        public Builder addProvisionTransfers(ProvisionTransferDto provisionTransferDto) {
            this.provisionTransfers.add(provisionTransferDto);
            return this;
        }

        public Builder addProvisionTransfers(Collection<? extends ProvisionTransferDto> provisionTransferDtos) {
            this.provisionTransfers.addAll(provisionTransferDtos);
            return this;
        }

        public Builder reserveTransfer(ReserveTransferDto reserveTransfer) {
            this.reserveTransfer = reserveTransfer;
            return this;
        }

        public TransferRequestParametersDto build() {
            return new TransferRequestParametersDto(quotaTransfers, reserveTransfer, provisionTransfers);
        }

    }

}
