package ru.yandex.intranet.d.web.model.transfers.api;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import io.swagger.v3.oas.annotations.media.Schema;

/**
 * Transfer request provider reserve responsible DTO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Schema(description = "Transfer request provider reserve responsible.")
@JsonIgnoreProperties(ignoreUnknown = true)
public final class TransferRequestProviderReserveResponsibleDto {

    private final List<String> responsibleUsers;
    private final String providerId;
    private final String folderId;
    private final long serviceId;

    @JsonCreator
    public TransferRequestProviderReserveResponsibleDto(List<String> responsibleUsers,
                                                        String providerId,
                                                        String folderId,
                                                        long serviceId) {
        this.responsibleUsers = responsibleUsers;
        this.providerId = providerId;
        this.folderId = folderId;
        this.serviceId = serviceId;
    }

    public static Builder builder() {
        return new Builder();
    }

    @Schema(description = "Responsible user uids.", required = true)
    public List<String> getResponsibleUsers() {
        return responsibleUsers;
    }

    @Schema(description = "Provider id.", required = true)
    public String getProviderId() {
        return providerId;
    }

    @Schema(description = "Reserve folder id.", required = true)
    public String getFolderId() {
        return folderId;
    }

    @Schema(description = "Provider service id.", required = true)
    public long getServiceId() {
        return serviceId;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        TransferRequestProviderReserveResponsibleDto that = (TransferRequestProviderReserveResponsibleDto) o;
        return serviceId == that.serviceId &&
                Objects.equals(responsibleUsers, that.responsibleUsers) &&
                Objects.equals(providerId, that.providerId) &&
                Objects.equals(folderId, that.folderId);
    }

    @Override
    public int hashCode() {
        return Objects.hash(responsibleUsers, providerId, folderId, serviceId);
    }

    @Override
    public String toString() {
        return "TransferRequestProviderReserveResponsibleDto{" +
                "responsibleUsers=" + responsibleUsers +
                ", providerId='" + providerId + '\'' +
                ", folderId='" + folderId + '\'' +
                ", serviceId=" + serviceId +
                '}';
    }

    public static final class Builder {

        private final List<String> responsibleUsers = new ArrayList<>();

        private String providerId;
        private String folderId;
        private Long serviceId;

        private Builder() {
        }

        public Builder providerId(String providerId) {
            this.providerId = providerId;
            return this;
        }

        public Builder folderId(String folderId) {
            this.folderId = folderId;
            return this;
        }

        public Builder serviceId(long serviceId) {
            this.serviceId = serviceId;
            return this;
        }

        public Builder addResponsibleUser(String responsibleUser) {
            this.responsibleUsers.add(responsibleUser);
            return this;
        }

        public Builder addResponsibleUsers(String... responsibleUsers) {
            this.responsibleUsers.addAll(Arrays.asList(responsibleUsers));
            return this;
        }

        public Builder addResponsibleUsers(Collection<? extends String> responsibleUsers) {
            this.responsibleUsers.addAll(responsibleUsers);
            return this;
        }

        public TransferRequestProviderReserveResponsibleDto build() {
            Objects.requireNonNull(providerId, "ProviderId is required");
            Objects.requireNonNull(folderId, "FolderId is required");
            Objects.requireNonNull(serviceId, "ServiceId is required");
            return new TransferRequestProviderReserveResponsibleDto(responsibleUsers, providerId, folderId, serviceId);
        }
    }

}
