package ru.yandex.intranet.d.web.model.transfers.api;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import io.swagger.v3.oas.annotations.media.Schema;

/**
 * Transfer request provider super responsible DTO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Schema(description = "Transfer request provider super responsible.")
@JsonIgnoreProperties(ignoreUnknown = true)
public final class TransferRequestProviderSuperResponsibleDto {

    private final String responsible;
    private final List<String> providerIds;

    @JsonCreator
    public TransferRequestProviderSuperResponsibleDto(String responsible, List<String> providerIds) {
        this.responsible = responsible;
        this.providerIds = providerIds;
    }

    public static Builder builder() {
        return new Builder();
    }

    @Schema(description = "Responsible user uid.", required = true)
    public String getResponsible() {
        return responsible;
    }

    @Schema(description = "Matching provider ids.", required = true)
    public List<String> getProviderIds() {
        return providerIds;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        TransferRequestProviderSuperResponsibleDto that = (TransferRequestProviderSuperResponsibleDto) o;
        return Objects.equals(responsible, that.responsible) &&
                Objects.equals(providerIds, that.providerIds);
    }

    @Override
    public int hashCode() {
        return Objects.hash(responsible, providerIds);
    }

    @Override
    public String toString() {
        return "TransferRequestProviderSuperResponsibleDto{" +
                "responsible='" + responsible + '\'' +
                ", providerIds=" + providerIds +
                '}';
    }

    public static final class Builder {

        private final List<String> providerIds = new ArrayList<>();

        private String responsible;

        private Builder() {
        }

        public Builder responsible(String responsible) {
            this.responsible = responsible;
            return this;
        }

        public Builder addProviderId(String providerId) {
            this.providerIds.add(providerId);
            return this;
        }

        public Builder addProviderIds(String... providerIds) {
            this.providerIds.addAll(Arrays.asList(providerIds));
            return this;
        }

        public Builder addProviderIds(Collection<? extends String> providerIds) {
            this.providerIds.addAll(providerIds);
            return this;
        }

        public TransferRequestProviderSuperResponsibleDto build() {
            Objects.requireNonNull(responsible, "Responsible is required");
            return new TransferRequestProviderSuperResponsibleDto(responsible, providerIds);
        }

    }

}
