package ru.yandex.intranet.d.web.model.transfers.api;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import io.swagger.v3.oas.annotations.media.Schema;

/**
 * Transfer request responsible DTO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Schema(description = "Transfer request responsible.")
@JsonIgnoreProperties(ignoreUnknown = true)
public final class TransferRequestResponsibleDto {

    private final List<TransferRequestResponsibleGroupDto> grouped;
    private final List<TransferRequestProviderSuperResponsibleDto> providerSuperResponsible;
    private final List<TransferRequestProviderReserveResponsibleDto> providerReserveResponsible;

    @JsonCreator
    public TransferRequestResponsibleDto(
            List<TransferRequestResponsibleGroupDto> grouped,
            List<TransferRequestProviderSuperResponsibleDto> providerSuperResponsible,
            List<TransferRequestProviderReserveResponsibleDto> providerReserveResponsible) {
        this.grouped = grouped != null ? grouped : List.of();
        this.providerSuperResponsible = providerSuperResponsible != null ? providerSuperResponsible : List.of();
        this.providerReserveResponsible = providerReserveResponsible != null ? providerReserveResponsible : List.of();
    }

    public static Builder builder() {
        return new Builder();
    }

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    @Schema(description = "Grouped transfer request responsible users.")
    public List<TransferRequestResponsibleGroupDto> getGrouped() {
        return grouped;
    }

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    @Schema(description = "Transfer request super responsible users for providers.")
    public List<TransferRequestProviderSuperResponsibleDto> getProviderSuperResponsible() {
        return providerSuperResponsible;
    }

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    @Schema(description = "Transfer request responsible users for provider reserve transfers.")
    public List<TransferRequestProviderReserveResponsibleDto> getProviderReserveResponsible() {
        return providerReserveResponsible;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        TransferRequestResponsibleDto that = (TransferRequestResponsibleDto) o;
        return Objects.equals(grouped, that.grouped) &&
                Objects.equals(providerSuperResponsible, that.providerSuperResponsible) &&
                Objects.equals(providerReserveResponsible, that.providerReserveResponsible);
    }

    @Override
    public int hashCode() {
        return Objects.hash(grouped, providerSuperResponsible, providerReserveResponsible);
    }

    @Override
    public String toString() {
        return "TransferRequestResponsibleDto{" +
                "grouped=" + grouped +
                ", providerSuperResponsible=" + providerSuperResponsible +
                ", providerReserveResponsible=" + providerReserveResponsible +
                '}';
    }

    public static final class Builder {

        private final List<TransferRequestResponsibleGroupDto> grouped = new ArrayList<>();
        private final List<TransferRequestProviderSuperResponsibleDto> providerSuperResponsible = new ArrayList<>();
        private final List<TransferRequestProviderReserveResponsibleDto> providerReserveResponsible = new ArrayList<>();

        private Builder() {
        }

        public Builder addGroup(TransferRequestResponsibleGroupDto group) {
            this.grouped.add(group);
            return this;
        }

        public Builder addGroups(TransferRequestResponsibleGroupDto... groups) {
            this.grouped.addAll(Arrays.asList(groups));
            return this;
        }

        public Builder addGroups(Collection<? extends TransferRequestResponsibleGroupDto> groups) {
            this.grouped.addAll(groups);
            return this;
        }

        public Builder addProviderSuperResponsibleUser(
                TransferRequestProviderSuperResponsibleDto providerSuperResponsibleUser) {
            this.providerSuperResponsible.add(providerSuperResponsibleUser);
            return this;
        }

        public Builder addProviderSuperResponsibleUsers(
                TransferRequestProviderSuperResponsibleDto... providerSuperResponsibleUsers) {
            this.providerSuperResponsible.addAll(Arrays.asList(providerSuperResponsibleUsers));
            return this;
        }

        public Builder addProviderSuperResponsibleUsers(
                Collection<? extends TransferRequestProviderSuperResponsibleDto> providerSuperResponsibleUsers) {
            this.providerSuperResponsible.addAll(providerSuperResponsibleUsers);
            return this;
        }

        public Builder addProviderReserveResponsibleUser(
                TransferRequestProviderReserveResponsibleDto providerReserveResponsibleUser) {
            this.providerReserveResponsible.add(providerReserveResponsibleUser);
            return this;
        }

        public Builder addProviderReserveResponsibleUsers(
                TransferRequestProviderReserveResponsibleDto... providerReserveResponsibleUsers) {
            this.providerReserveResponsible.addAll(Arrays.asList(providerReserveResponsibleUsers));
            return this;
        }

        public Builder addProviderReserveResponsibleUsers(
                Collection<? extends TransferRequestProviderReserveResponsibleDto> providerReserveResponsibleUsers) {
            this.providerReserveResponsible.addAll(providerReserveResponsibleUsers);
            return this;
        }

        public TransferRequestResponsibleDto build() {
            return new TransferRequestResponsibleDto(grouped, providerSuperResponsible, providerReserveResponsible);
        }

    }

}
