package ru.yandex.intranet.d.web.model.transfers.api;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import io.swagger.v3.oas.annotations.media.Schema;

/**
 * Transfer request responsible group DTO.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Schema(description = "Transfer request responsible group.")
@JsonIgnoreProperties(ignoreUnknown = true)
public final class TransferRequestResponsibleGroupDto {

    private final List<String> folderIds;
    private final List<TransferRequestResponsibleUserDto> responsibleSet;

    @JsonCreator
    public TransferRequestResponsibleGroupDto(List<String> folderIds,
                                              List<TransferRequestResponsibleUserDto> responsibleSet) {
        this.folderIds = folderIds;
        this.responsibleSet = responsibleSet;
    }

    public static Builder builder() {
        return new Builder();
    }

    @Schema(description = "Folder ids.", required = true)
    public List<String> getFolderIds() {
        return folderIds;
    }

    @Schema(description = "Responsible users with matching services.", required = true)
    public List<TransferRequestResponsibleUserDto> getResponsibleSet() {
        return responsibleSet;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        TransferRequestResponsibleGroupDto that = (TransferRequestResponsibleGroupDto) o;
        return Objects.equals(folderIds, that.folderIds) &&
                Objects.equals(responsibleSet, that.responsibleSet);
    }

    @Override
    public int hashCode() {
        return Objects.hash(folderIds, responsibleSet);
    }

    @Override
    public String toString() {
        return "TransferRequestResponsibleGroupDto{" +
                "folderIds=" + folderIds +
                ", responsibleSet=" + responsibleSet +
                '}';
    }

    public static final class Builder {

        private final List<String> folderIds = new ArrayList<>();
        private final List<TransferRequestResponsibleUserDto> responsibleSet = new ArrayList<>();

        private Builder() {
        }

        public Builder addFolderId(String folderId) {
            this.folderIds.add(folderId);
            return this;
        }

        public Builder addFolderIds(String... folderIds) {
            this.folderIds.addAll(Arrays.asList(folderIds));
            return this;
        }

        public Builder addFolderIds(Collection<? extends String> folderIds) {
            this.folderIds.addAll(folderIds);
            return this;
        }

        public Builder addResponsibleUser(TransferRequestResponsibleUserDto responsibleUser) {
            this.responsibleSet.add(responsibleUser);
            return this;
        }

        public Builder addResponsibleUsers(TransferRequestResponsibleUserDto... responsibleUsers) {
            this.responsibleSet.addAll(Arrays.asList(responsibleUsers));
            return this;
        }

        public Builder addResponsibleUsers(Collection<? extends TransferRequestResponsibleUserDto> responsibleUsers) {
            this.responsibleSet.addAll(responsibleUsers);
            return this;
        }

        public TransferRequestResponsibleGroupDto build() {
            return new TransferRequestResponsibleGroupDto(folderIds, responsibleSet);
        }

    }

}
